import React, { useEffect, useMemo } from 'react';
import PropTypes from 'prop-types';
import get from 'lodash/get';
import WordBreak from 'components/WordBreak';
import createI18N from '@yandex-int/i18n';
import { Block as Account } from 'pages/mail/modules/account';
import { Opportunities, OpportunitiesServiceImpl } from 'pages/mail/modules/opportunities';
import Panel from 'modules/panel';
import { ContactCreateModal } from 'modules/contactCreateForm';
import cx from 'classnames';
import Requests from 'pages/mail/modules/requests';
import { PanelFiles, PanelEntity } from 'modules/clientHistory';
import Mail from 'components/Mail';
import * as keyset from './MailPreview.i18n';
import css from './styles.modules.scss';
import fromPopupDecorator from './fromPopupDecorator';
import MailComment from './MailComment';
import Header from './Header';
import Actions from '../Actions/Actions';
import SelectContactModal from './selectContactModal';

const i18n = createI18N(keyset);
const i18nAttributes = i18n('attributes');
const i18nAccountHistory = i18n('accountHistory');
const i18nAccountFiles = i18n('accountFiles');

const openCreateContact = (form, contact, account) => {
  if (!contact || !account) {
    return;
  }

  const emails = [];

  if (contact.email) {
    emails.push({ email: contact.email });
  }

  ContactCreateModal.open({
    form,
    accountId: account.id,
    isNew: true,
    initialValues: {
      firstName: contact.displayName,
      emails,
    },
  });
};

const MailBlock = ({ className, mainClassName, readonly, sidebar, children, ...props }) => {
  const opportunitiesService = useMemo(() => new OpportunitiesServiceImpl(props.mailId), []);

  useEffect(() => {
    if (props.isLoad) {
      opportunitiesService.setMailId(props.mailId);
      opportunitiesService.setOpportunities(props.data?.opportunities || []);
    }
  }, [props.isLoad, props.mailId]);

  if (props.isLoad && opportunitiesService) {
    const contact = get(props, 'data.mail.contact');
    const addContactFormName = 'fromPopup:contacts:NEW';
    const selectContactFormName = 'fromPopup:contacts:SELECT';
    const fromDecorator =
      contact && contact.showPopup && props.data.account
        ? fromPopupDecorator({
            mailId: props.data.mail.id,
            modalFormNameSelect: selectContactFormName,
            onCreateContact: () => {
              openCreateContact(addContactFormName, contact, props.data.account);
            },
          })
        : null;
    const account = props.data && props.data.account;
    const accountId = account && account.id;
    let panels = [];

    panels.push({
      id: 'attributes',
      title: i18nAttributes,
      content: (
        <WordBreak className={css.side}>
          <Account
            className={css.block}
            account={props.data.account}
            agency={props.agency}
            onChange={props.setAccount}
            headerClassName={css.block__title}
            actionClassName={css.block__action}
            bodyClassName={css.block__body}
            readonly={readonly}
          />
          <Requests
            mailId={props.data.mail.id}
            className={css.block}
            headerClassName={css.block__title}
            actionClassName={css.block__action}
            bodyClassName={css.block__body}
            items={props.data.requests}
            readonly={readonly}
            addRequest={props.addRequest}
          />
          <Opportunities
            account={account}
            mailId={props.data.mail.id}
            className={css.block}
            headerClassName={css.block__title}
            actionClassName={css.block__action}
            bodyClassName={css.block__body}
            opportunitiesService={opportunitiesService}
            readonly={readonly}
          />
        </WordBreak>
      ),
    });

    if (accountId) {
      panels = panels.concat([
        {
          id: 'activity',
          title: i18nAccountHistory,
          content: (
            <div className={css.clientHistoryWrap}>
              <PanelEntity
                key={accountId}
                param={{ clientId: accountId }}
                name="activity"
                loadOnMount
              />
            </div>
          ),
          keepMount: true,
          padding: false,
        },
        {
          id: 'files',
          title: i18nAccountFiles,
          content: (
            <div className={css.clientHistoryWrap}>
              <PanelFiles
                key={accountId}
                param={{ clientId: accountId }}
                name="files"
                loadOnMount
                showTitle={false}
              />
            </div>
          ),
          keepMount: true,
          padding: false,
        },
      ]);
    }

    return (
      <div className={cx(css.root, className)} key={props.data.mail.id}>
        <div className={cx(css.main, mainClassName)}>
          {!readonly && (
            <Actions
              mailId={props.data.mail.id}
              onReply={props.onReply}
              onReplyAll={props.onReplyAll}
              onForward={props.onForward}
              onRemove={props.remove}
              onDraft={props.onDraft}
              onSpam={props.spam}
              onUnread={props.unread}
              isUnread={props.data.mail.isUnread}
              isSpam={props.data.mail.isSpam}
              type={props.data.mail.type}
            />
          )}
          <Mail
            className={css.mail}
            theme="MAIL_MODULE"
            header={Header}
            {...props.data.mail}
            requests={props.data.requests}
            fromDecoratorFunc={fromDecorator}
            showMeta
            showActions={!readonly}
            readonly={readonly}
          />
          <SelectContactModal
            formName={selectContactFormName}
            account={props.data.account}
            contact={contact}
          />
          <MailComment
            mailId={props.data.mail.id}
            comment={props.data.comment}
            className={css.comment}
            onSave={props.commentSave}
            onRemove={props.commentRemove}
            readonly={readonly}
          />
        </div>
        {sidebar && <Panel name="mail/panel" position="right" panels={panels} />}
        {children}
      </div>
    );
  }

  return null;
};

MailBlock.propTypes = {
  className: PropTypes.string,
  mainClassName: PropTypes.string,
  readonly: PropTypes.bool,
  sidebar: PropTypes.bool,
  children: PropTypes.oneOfType([PropTypes.element, PropTypes.arrayOf(PropTypes.element)]),
  isLoad: PropTypes.bool,
  data: PropTypes.shape({
    mail: PropTypes.shape({
      id: PropTypes.number,
      contact: PropTypes.shape({
        showPopup: PropTypes.bool,
      }),
      isUnread: PropTypes.bool,
      isSpam: PropTypes.bool,
      type: PropTypes.string,
    }),
    account: PropTypes.shape({
      id: PropTypes.number,
    }),
    requests: PropTypes.arrayOf(PropTypes.instanceOf(Object)),
    opportunities: PropTypes.arrayOf(PropTypes.instanceOf(Object)),
    comment: PropTypes.string,
  }).isRequired,
  agency: PropTypes.instanceOf(Object),
  onReply: PropTypes.func.isRequired,
  onReplyAll: PropTypes.func.isRequired,
  onForward: PropTypes.func.isRequired,
  onDraft: PropTypes.func.isRequired,
  remove: PropTypes.func.isRequired,
  spam: PropTypes.func.isRequired,
  unread: PropTypes.func.isRequired,
  commentSave: PropTypes.func.isRequired,
  commentRemove: PropTypes.func.isRequired,
  setAccount: PropTypes.func.isRequired,
  addRequest: PropTypes.func.isRequired,
};

MailBlock.defaultProps = {
  className: undefined,
  mainClassName: undefined,
  readonly: undefined,
  sidebar: true,
  children: undefined,
  isLoad: undefined,
  agency: {},
};

export default MailBlock;
