import React, { FC } from 'react';
import { observer } from 'mobx-react-lite';
import { EType } from 'types/entities';
import { buildUrl } from 'utils/buildUrl';
import Button from '@crm/components/dist/lego2/Button';
import createI18N from '@yandex-int/i18n';
import { ModalGrid } from 'components/SuggestGrid/ModalGrid/ModalGrid';
import { Opportunity } from './Opportunity';
import { OpportunitiesProps } from './Opportunities.types';
import * as keyset from './opportunities.i18n';

const i18n = createI18N(keyset);
const i18nLink = i18n('link');
const i18nEdit = i18n('edit');
const i18nOpportunities = i18n('opportunities');

export const Opportunities = observer((({
  className,
  headerClassName,
  bodyClassName,
  actionClassName,
  itemClassName,
  opportunitiesService,
  readonly,
  account,
  mailId,
}) => {
  const [visible, setVisible] = React.useState(false);

  const openModal = () => setVisible(true);
  const closeModal = () => setVisible(false);

  const handleSave = (data) => {
    opportunitiesService.save(data);
    closeModal();
  };

  const providerUrl = buildUrl('/v0/blocks/opportunity/suggest', {
    accountId: account?.id,
    eType: EType.Mail,
    eId: mailId,
  });

  return (
    <div className={className}>
      <span className={headerClassName}>
        {i18nOpportunities}
        {!readonly && account?.id && (
          <span className={actionClassName}>
            <Button size="xs" view="pseudo" onClick={openModal}>
              {opportunitiesService.opportunities.length ? i18nEdit : i18nLink}
            </Button>
            <ModalGrid
              visible={visible}
              selected={opportunitiesService.opportunities}
              onClose={closeModal}
              onSave={handleSave}
              provider={providerUrl}
              account={account}
            />
          </span>
        )}
      </span>
      {Boolean(opportunitiesService.opportunities.length) && (
        <div className={bodyClassName}>
          {opportunitiesService.opportunities.map((item) => (
            <Opportunity key={item.id} className={itemClassName} opportunity={item} />
          ))}
        </div>
      )}
    </div>
  );
}) as FC<OpportunitiesProps>);
