import React from 'react';
import PropTypes from 'prop-types';
import get from 'lodash/get';
import 'styles/component/modalField.scss';
import Icon from '@crm/components/dist/lego2/Icon';
import Button from '@crm/components/dist/lego2/Button/Button';
import { unv } from 'utils/common';

const propTypes = {
  preview: PropTypes.element,
  modal: PropTypes.element,
  showPreview: PropTypes.bool,
  onChangeValue: PropTypes.string,
  clear: PropTypes.bool,
  apiV2: PropTypes.bool, // новый api модальных окон
};

const defaultProps = {
  showPreview: true,
  onChangeValue: 'item', // or 'id',
  clear: false,
  apiV2: false,
};

export default class ModalField extends React.Component {
  constructor(props) {
    super(props);

    this.state = {
      show: false,
    };
  }

  handleToggle = (flag) => {
    if (this.props.apiV2) {
      this.modal.open();
    } else {
      this.setState({ show: unv(flag, !this.state.show) });
    }
  };

  handleOnChange = (id, item) => {
    if (this.props.apiV2 || this.props.onChangeValue === 'id') {
      this.props.input.onChange(id);
    } else {
      this.props.input.onChange(item);
    }
  };

  getSelectedId() {
    if (this.props.onChangeValue === 'id') {
      return get(this.props, 'input.value');
    }
    return get(this.props, 'input.value.id');
  }

  renderModal() {
    const Modal = this.props.modal;

    if (this.props.apiV2) {
      return (
        <Modal
          name={this.props.input.name}
          ref={(modal) => {
            this.modal = modal;
          }}
          onSubmit={this.handleOnChange}
        />
      );
    }

    /* При закрытии модального окна, размонтируем его, чтобы при открытии он мог вызвать didMount
     * и загрузить данные с бэка
     * Проблема: при закрытии окна не происходит плавной анимации
     * */
    if (this.state.show) {
      return (
        <Modal
          show={this.state.show}
          toggle={this.handleToggle}
          onChange={this.handleOnChange}
          destroyOnUnmount
          selectedId={this.getSelectedId()}
          {...this.props.modalProps}
        />
      );
    }

    return null;
  }

  render() {
    const Preview = this.props.preview;
    const Modal = this.props.modal;
    const { showPreview, clear } = this.props;

    const renderPreview = () => {
      if (typeof this.props.preview === 'string') {
        return this.props.preview;
      }

      if (!this.props.input.value) {
        return 'не выбрано';
      }

      if (!showPreview) {
        return null;
      }

      return Preview ? <Preview {...this.props.input.value} /> : this.props.input.value;
    };

    return (
      <span className={this.props.className}>
        <a className="modalField" tabIndex="0" onClick={this.handleToggle}>
          {renderPreview()}
        </a>
        {clear && this.props.input.value && (
          <Button
            view="plain"
            size="xs"
            tabIndex="0"
            title="Очистить"
            onClick={() => this.props.input.onChange('')}
            icon={(className) => <Icon className={className} type="close" />}
          />
        )}
        {Modal ? this.renderModal() : null}
      </span>
    );
  }
}

ModalField.propTypes = propTypes;
ModalField.defaultProps = defaultProps;
