import { AdaptiveIntervalOptions } from './AdaptiveIntervalOptions';

export class AdaptiveInterval {
  public static readonly MULTIPLIER_FOR_BACKGROUND_TASK = 2;

  public static formatOptions(optionsOrTimeout: AdaptiveIntervalOptions | number) {
    return typeof optionsOrTimeout === 'number'
      ? { timeoutNormal: optionsOrTimeout }
      : optionsOrTimeout;
  }

  private readonly handler: Function;
  private readonly options: AdaptiveIntervalOptions;
  private readonly arguments: unknown[];

  private timer?: number;
  private timeout?: number;

  constructor(
    handler: Function,
    optionsOrTimeout: AdaptiveIntervalOptions | number,
    ...args: unknown[]
  ) {
    this.handler = handler;
    this.arguments = args;
    this.options = AdaptiveInterval.formatOptions(optionsOrTimeout);
  }

  runHandler() {
    this.handler(...this.arguments);
  }

  setNormalInterval() {
    this.clearInterval();
    this.timeout = this.options.timeoutNormal;
    this.runInterval();
  }

  setSlowInterval() {
    this.clearInterval();
    this.timeout =
      this.options.timeoutSlow ??
      this.options.timeoutNormal * AdaptiveInterval.MULTIPLIER_FOR_BACKGROUND_TASK;
    this.runInterval();
  }

  clearInterval() {
    window.clearInterval(this.timer);
    this.timer = undefined;
  }

  private runInterval() {
    if (!this.canRunInterval()) {
      return;
    }

    this.timer = window.setInterval(this.handler, this.timeout, ...this.arguments);
  }

  private canRunInterval(): boolean {
    return Boolean(this.timeout && this.timeout !== 0);
  }
}
