import { reaction, IReactionDisposer } from 'mobx';
import debounce from 'lodash/debounce';
import { Cancelable } from 'lodash';
import { skipCallWithSameArgs } from 'utils/skipCallWithSameArgs';
import { AdaptiveInterval } from './AdaptiveInterval';
import { AdaptiveIntervalOptions } from './AdaptiveIntervalOptions';

export interface CreateAdaptiveIntervalByMobxClassOptions {
  switcher: () => boolean;
  debounceDelay?: number;
  adaptiveIntervalOptions?: Partial<AdaptiveIntervalOptions>;
}

export const createAdaptiveIntervalByMobxClass = (
  options: CreateAdaptiveIntervalByMobxClassOptions,
) => {
  options = { ...{ debounceDelay: 0 }, ...options };

  return class AdaptiveIntervalByMobx {
    public static DEBOUNCE_DELAY = options.debounceDelay;

    private reactionDisposer: IReactionDisposer;

    private isAutorunCalled = false;

    private readonly switchIntervalDebounce: ((isUseNormalInterval: boolean) => void) & Cancelable;

    private readonly adaptiveTimer: AdaptiveInterval;

    constructor(
      handler: Function,
      optionsOrTimeout: AdaptiveIntervalOptions | number,
      ...args: unknown[]
    ) {
      const formattedOptions = {
        ...options.adaptiveIntervalOptions,
        ...AdaptiveInterval.formatOptions(optionsOrTimeout),
      };

      this.switchInterval = this.switchInterval.bind(this);
      this.switchIntervalDebounce = debounce(
        skipCallWithSameArgs(this.switchInterval),
        AdaptiveIntervalByMobx.DEBOUNCE_DELAY,
      );

      this.adaptiveTimer = new AdaptiveInterval(handler, formattedOptions, ...args);

      this.init();
    }

    destroy() {
      this.reactionDisposer();
      this.switchIntervalDebounce.cancel();
      this.adaptiveTimer.clearInterval();
    }

    private switchInterval(isUseNormalInterval) {
      if (isUseNormalInterval) {
        if (this.isAutorunCalled) {
          this.adaptiveTimer.runHandler();
        }
        this.adaptiveTimer.setNormalInterval();
      } else {
        this.adaptiveTimer.setSlowInterval();
      }

      this.isAutorunCalled = true;
    }

    private init() {
      this.reactionDisposer = reaction(() => options.switcher(), this.switchIntervalDebounce, {
        fireImmediately: true,
      });
    }
  };
};
