import { Subject, merge, Observable, SubscribableOrPromise, timer, from, Subscription } from 'rxjs';
import { filter, map, mapTo, take, takeUntil, debounceTime, scan } from 'rxjs/operators';
import { BatchTask as IBatchTask } from 'types/BatchTask';

export interface BatchTaskOptions {
  totalSubTasks?: number;
  debounceTime?: number;
}

export class BatchTask implements IBatchTask {
  static readonly DEFAULT_DEBOUNCE_TIME = 500;

  private options: Required<BatchTaskOptions>;

  private totalTasks = new Subject<number>();
  private taskComplete = new Subject<boolean>();
  private complete = new Subject<void>();
  private completeSubscription = new Subscription();

  public asObservable: Observable<void>;

  private isCompleted = false;

  private taskToSubscription = new Map<SubscribableOrPromise<unknown>, Subscription>();

  constructor(options?: BatchTaskOptions) {
    this.setOptions(options);
    this.asObservable = this.complete.asObservable();

    if (this.options.totalSubTasks === 0) {
      this.createObservableWithZeroSubTasks();
      return;
    }

    if (this.options.totalSubTasks > 0) {
      this.createObservableWithLimitedSubTasks();
      return;
    }

    this.createObservableWithUnknownSubTasks();
  }

  destroy(error?: Error) {
    if (this.isCompleted) {
      return;
    }

    this.isCompleted = true;
    this.taskToSubscription.forEach((subscription) => subscription.unsubscribe());
    this.taskToSubscription.clear();

    this.completeSubscription.unsubscribe();

    if (error) {
      this.complete.error(error);

      return;
    }

    this.complete.complete();
  }

  registerTask(observable: SubscribableOrPromise<unknown>) {
    if (this.isCompleted) {
      return;
    }

    this.totalTasks.next(this.taskToSubscription.size + 1);
    this.subscribeToChildTask(observable);
  }

  unregisterTask(observable: SubscribableOrPromise<unknown>) {
    if (this.isCompleted) {
      return;
    }

    const subscription = this.taskToSubscription.get(observable);
    subscription?.unsubscribe();
    this.taskToSubscription.delete(observable);

    this.totalTasks.next(this.taskToSubscription.size);
  }

  private createObservableWithZeroSubTasks() {
    this.complete.complete();
    this.isCompleted = true;
  }

  private createObservableWithLimitedSubTasks() {
    this.subscribeToCompleteLogic(
      this.taskComplete.pipe(
        scan((acc) => acc + 1, 0),
        filter((count) => count === this.options.totalSubTasks),
        mapTo(true),
      ),
    );
  }

  private createObservableWithUnknownSubTasks() {
    this.subscribeToCompleteLogic(
      merge(
        this.totalTasks.pipe(
          debounceTime(this.options.debounceTime),
          map((value) => value === 0),
        ),
        timer(this.options.debounceTime).pipe(
          mapTo(true),
          takeUntil(this.totalTasks.pipe(filter((value) => value !== 0))),
        ),
      ),
    );
  }

  private subscribeToCompleteLogic(observable: Observable<boolean>) {
    this.completeSubscription.add(
      observable.pipe(filter(Boolean), take(1)).subscribe(this.complete),
    );
  }

  private setOptions(options?: BatchTaskOptions) {
    this.options = {
      totalSubTasks: options?.totalSubTasks ?? -1,
      debounceTime: options?.debounceTime ?? BatchTask.DEFAULT_DEBOUNCE_TIME,
    };
  }

  private subscribeToChildTask(observable: SubscribableOrPromise<unknown>) {
    this.taskToSubscription.set(
      observable,
      from(observable).subscribe({
        error: (error) => {
          this.destroy(error);
        },
        complete: () => {
          this.unregisterTask(observable);
          this.taskComplete.next(true);
        },
      }),
    );
  }
}
