import { History } from 'history';
import { CRMXiva, XivaBackendEventType, CRMXivaCustomEvent } from 'modules/xiva';
import { IssueCallNewNotifyData } from 'types/xiva/IssueCallNewNotify';
import { activeCallIssueService } from 'services/ActiveCallIssueService';
import { showPinModalIfNeeded } from 'modules/pinForm';
import { TabMutex } from 'utils/TabMutex';
import createLink from 'modules/issues/utils/createLink';
import browserWindow from 'utils/browserWindow';
import { terminalIdService } from '../TerminalIdService';

export class CallIssueRedirectByXiva {
  private static createIssueLink(issue: IssueCallNewNotifyData['issue']): string {
    return createLink({
      id: issue.id,
      typeId: issue.typeId,
      hash: false,
    });
  }

  constructor(private history: History, private xiva: CRMXiva) {
    this.handleNewIssueCall = this.handleNewIssueCall.bind(this);

    this.init();
  }

  destroy() {
    this.xiva.removeEventListener(XivaBackendEventType.IssueCallNewNotify, this.handleNewIssueCall);
  }

  private init() {
    this.xiva.addEventListener(XivaBackendEventType.IssueCallNewNotify, this.handleNewIssueCall);
  }

  private handleNewIssueCall(event: CRMXivaCustomEvent<IssueCallNewNotifyData>) {
    if (!event.detail.terminalId) {
      this.openIssueInNewTab(event.detail);
      return;
    }

    if (event.detail.terminalId === terminalIdService.value) {
      activeCallIssueService.issue = event.detail.issue;
      this.openIssueInThisTab(event.detail);
    }
  }

  private openIssueInThisTab(data: IssueCallNewNotifyData) {
    this.history.push(CallIssueRedirectByXiva.createIssueLink(data.issue));
    if (data.autoShowPinFormConfig) {
      showPinModalIfNeeded(data.autoShowPinFormConfig);
    }
  }

  private openIssueInNewTab(data: IssueCallNewNotifyData) {
    TabMutex.createAndRun({
      id: `issue_${data.issue.id}_open`,
      task: () => {
        const childWindow = browserWindow.open(CallIssueRedirectByXiva.createIssueLink(data.issue));
        if (data.autoShowPinFormConfig) {
          childWindow.pinConfig = data.autoShowPinFormConfig;
        }
      },
    });
  }
}
