import UAParser from 'ua-parser-js';
import { MessageLevel } from 'types/Message/MessageLevel';
import { createSharedWorker } from 'utils/createSharedWorker';
import { logger } from 'services/Logger';
import { Log } from 'services/Log';
import { push, close } from 'modules/notificationsUI/actions';

export class ConnectionCheckerLog extends Log {
  constructor(message) {
    super(message);
    this.name = `ConnectionChecker:${message}`;
  }
}

const HEALTHCHECK_URL = `${window.location.origin}/ready`;
const DISCONNECT_MESSAGE = 'Пропало соединение с сервисами CRM';
const CONNECT_MESSAGE = 'Соединение с сервисами CRM восстановлено';

type StatusType = 'online' | 'offline';

export class ConnectionChecker {
  private worker: SharedWorker;
  private status: StatusType = 'online';
  private id: string;
  private readonly instance: ConnectionChecker;

  constructor() {
    if (this.instance) {
      return this.instance;
    }
    this.id = Date.now().toString();
    this.instance = this;
  }

  public run = () => {
    const uaParser = new UAParser();
    const { name } = uaParser.getBrowser();
    if (name !== 'Safari') {
      try {
        this.worker = createSharedWorker('connectionChecker', 'CRMconnectionChecker');
        this.worker.port.addEventListener('message', this.messageHandler, false);
        this.worker.port.start();
        this.worker.port.postMessage(['connect', HEALTHCHECK_URL]);
      } catch (error) {
        logger.reportError(error);
      }
    }
  };

  public messageHandler = (e) => {
    if (e?.data?.status && e?.data?.status !== this.status) {
      this.status = e?.data?.status;
      if (this.status === 'offline') {
        logger.reportInfo(new ConnectionCheckerLog('connection-lost'));
        window.reduxStore.dispatch(
          push({
            text: DISCONNECT_MESSAGE,
            level: MessageLevel.Error,
            id: this.id,
            autoHideDuration: 0,
            isPermanent: true,
          }),
        );
      } else {
        logger.reportInfo(new ConnectionCheckerLog('connection-is-established'));
        window.reduxStore.dispatch(close(this.id));
        window.reduxStore.dispatch(
          push({
            text: CONNECT_MESSAGE,
            level: MessageLevel.Success,
            autoHideDuration: 4000,
          }),
        );
      }
    }
  };

  public isOnline = () => {
    return this.status === 'online';
  };
}
