import { Store } from 'redux';
import { BackendError } from 'api/BackendError';
import { convertAnyToError } from 'services/Logger';
import UIBaseException from 'utils/exceptions/UIBaseException';

export class FrontendErrorWatchService {
  private readonly store: Store<unknown>;

  constructor(store: Store<unknown>) {
    this.store = store;

    window.addEventListener('unhandledrejection', this.handleUnhandledRejection);

    window.onerror = (errorMsg, url, lineNumber, colNumber, error) => {
      if (String.prototype.startsWith.call(errorMsg, 'ResizeObserver')) {
        return;
      }
      const errorObj = convertAnyToError(error);
      this.store.dispatch({
        type: 'CRASH_REPORT',
        url,
        line: lineNumber,
        errorObj,
        alertText: `Ошибка: ${errorMsg}:${url}:${lineNumber}`,
        alertType: 'ERROR',
      });
    };
  }

  destroy() {
    window.removeEventListener('unhandledrejection', this.handleUnhandledRejection);
  }

  private handleUnhandledRejection = (
    event: PromiseRejectionEvent | CustomEvent<{ reason: Error | string }>,
  ) => {
    let error;
    if ('reason' in event) {
      error = event.reason;
    } else {
      error = Object(event.detail).reason;
    }

    if (!error) {
      return;
    }

    if (error instanceof UIBaseException) {
      event.preventDefault();
      return;
    }

    if (error instanceof BackendError) {
      event.preventDefault();
      return;
    }
    const errorObj = convertAnyToError(error);
    this.store.dispatch({
      type: 'CRASH_REPORT',
      errorObj,
      alertText: errorObj.message,
      alertType: 'ERROR',
    });
  };
}
