import React, { useState, memo, FC } from 'react';
import { observer } from 'mobx-react-lite';
import InfiniteList from 'components/InfiniteList';
import { LoadState } from 'types/LoadState';
import { PageableItemDTO } from 'types/pagination/PageableItemDTO';
import { worker } from './mocks/browser';
import { GridServiceByUrl } from './GridService';
import { SortOrder } from './types/GridMeta';

export default {
  title: 'GridService',
  decorators: [
    (Story) => {
      worker?.start();
      worker?.resetHandlers();
      return <Story />;
    },
  ],
};

const InfiniteListItem = memo((props: { item: PageableItemDTO }) => {
  return <div style={{ height: 100, flexShrink: 0 }}>{props.item.id}</div>;
});

const SORT_TRANSITION_MAP = {
  [SortOrder.Asc]: SortOrder.Desc,
  [SortOrder.Desc]: SortOrder.Asc,
} as const;

const InfiniteListWrap: FC<{ url: string }> = observer(({ url }) => {
  const [service] = useState(() => new GridServiceByUrl(url));

  return (
    <div>
      <div>
        <button onClick={service.reload}>reload</button>
        <div>
          {service.sortStore.getOrderIds().map((sortItemId) => {
            const item = service.sortStore.getItemById(sortItemId);

            if (!item) {
              return null;
            }

            const handleClick = () => {
              service.sortStore.setItemById(sortItemId, {
                id: sortItemId,
                order: SORT_TRANSITION_MAP[item?.order],
              });
            };

            return (
              <button key={sortItemId} onClick={handleClick}>
                {item.id}: {item.order}
              </button>
            );
          })}
        </div>
      </div>
      <InfiniteList
        onLoad={service.loader.load}
        isEof={!service.loader.hasNextPage}
        isLoading={service.loader.meta.state === LoadState.Pending}
        isLoadingFirstPage={
          service.loader.meta.state === LoadState.Pending && !service.loader.isFirstPageLoaded
        }
        isError={Boolean(service.loader.meta.error)}
        style={{ height: 500 }}
      >
        {service.store.getOrderIds().map((id) => {
          const item = service.store.getItemById(id);

          if (!item) {
            return null;
          }

          return <InfiniteListItem key={id} item={item} />;
        })}
      </InfiniteList>
    </div>
  );
});

export const Default = () => <InfiniteListWrap url="/grid" />;

export const Error = () => <InfiniteListWrap url="/grid/error" />;
