import { makeObservable, observable, action, reaction, IReactionDisposer } from 'mobx';
import { get } from 'api/common';
import { ListStore } from 'types/ListStore';
import { PaginationLoader } from 'types/PaginationLoader';
import { PageableDTO } from 'types/pagination/PageableDTO';
import { Destroyable } from 'types/Destroyable';
import { PageableItemDTO } from 'types/pagination/PageableItemDTO';
import { WithGridMeta, Sort } from './types/GridMeta';
import { PaginationLoaderByCallback } from '../PaginationLoaderByCallback';
import { NormalizedListStore } from '../NormalizedListStore';

export class GridServiceByUrl<Item extends PageableItemDTO> implements Destroyable {
  public store: ListStore<Item>;
  public loader: PaginationLoader<PageableDTO<Item> & WithGridMeta>;
  public sortStore: ListStore<Sort>;
  public fieldsVisibilityStore: ListStore<{ id: string }>;

  private filterReactionDisposer?: IReactionDisposer;

  private loadUrl: string;

  constructor(private readonly baseLoadUrl: string) {
    this.loadUrl = baseLoadUrl;

    this.reload = this.reload.bind(this);
    this.load = this.load.bind(this);
    this.loadItems = this.loadItems.bind(this);
    this.createInitLoadUrl = this.createInitLoadUrl.bind(this);

    makeObservable<GridServiceByUrl<Item>, 'loadItems' | 'initFilters'>(this, {
      store: observable,
      loader: observable,
      reload: action.bound,
      initFilters: action.bound,
      loadItems: action,
    });

    this.init();
  }

  public reload() {
    this.cleanup();
    this.init();
  }

  public destroy() {
    this.cleanup();
  }

  private init() {
    this.store = new NormalizedListStore();
    this.sortStore = new NormalizedListStore();
    this.fieldsVisibilityStore = new NormalizedListStore();
    this.loader = new PaginationLoaderByCallback(this.load, this.loadItems);
    this.loader.load();
  }

  private load(nextUrl?: string) {
    return get<PageableDTO<Item> & WithGridMeta>({
      url: nextUrl || this.loadUrl,
    }).then((data) => {
      if (!nextUrl) {
        this.initFilters(data);
        this.subscribeToChangeFilter();
      }

      return data;
    });
  }

  private initFilters(data: WithGridMeta) {
    this.initSortStore(data);
    this.initFieldsVisibilityStore(data);
  }

  private subscribeToChangeFilter() {
    this.filterReactionDisposer = reaction(this.createInitLoadUrl, (filteredLoadUrl) => {
      this.loadUrl = filteredLoadUrl;
      this.reload();
    });
  }

  private createInitLoadUrl(): string {
    const params = new URLSearchParams();
    params.set('sort', this.stringifySort());
    params.set('fieldsVisibility', this.stringifyFieldsVisibility());

    return `${this.baseLoadUrl}?${params.toString()}`;
  }

  private stringifySort(): string {
    return JSON.stringify(this.sortStore.getOrderIds().map((id) => this.sortStore.getItemById(id)));
  }

  private stringifyFieldsVisibility(): string {
    return JSON.stringify(this.fieldsVisibilityStore.getOrderIds());
  }

  private initSortStore(data: WithGridMeta) {
    data.meta.sort.forEach((item) => {
      this.sortStore.addItemById(item.id, item);
    });
  }

  private initFieldsVisibilityStore(data: WithGridMeta) {
    data.meta.fieldsVisibility.forEach((item) => {
      this.fieldsVisibilityStore.addItemById(item, { id: item });
    });
  }

  private cleanup() {
    this.loader.destroy();

    if (this.filterReactionDisposer) {
      this.filterReactionDisposer();
    }
  }

  private loadItems(items: Item[]) {
    items.forEach((item) => {
      this.store.addItemById(item.id, item);
    });
  }
}
