import isEmpty from 'lodash/isEmpty';
import { sendIntersectionTimeMark } from './utils/sendIntresectionTimeMark';
import { ObserverCollection, Observer, Observable, Target } from './IntersectionWatcher.types';
import { intersectionObserverConfig } from './IntersectionWatcher.config';

export class IntersectionWatcher {
  private observerCollection: ObserverCollection = {};

  public findObservableInObserver = (
    observer: Observer | undefined,
    target: Target,
  ): Observable | undefined => {
    if (observer && observer.observableCollection.length > 0) {
      return observer.observableCollection.find((observable) => observable.target === target);
    }
  };

  public getObserverByInstance = (instance: IntersectionObserver): Observer | undefined => {
    if (!isEmpty(this.observerCollection)) {
      return Object.values(this.observerCollection).find(
        (observer) => observer.instance === instance,
      );
    }
  };

  public targetIsObservable = (target: Target): boolean => {
    return Boolean(
      Object(this.observerCollection)?.values?.some((observer) =>
        observer.observableCollection.some((observable) => observable.target === target),
      ),
    );
  };

  public getObserverByName = (observerName: string, needCreate = false) => {
    if (!this.observerCollection[observerName] && needCreate) {
      this.observerCollection[observerName] = {
        name: observerName,
        instance: new IntersectionObserver(this.handleIntersection, intersectionObserverConfig),
        observableCollection: [],
      };
    }
    return this.observerCollection[observerName];
  };

  private handleIntersection = (entries, observerInstance) => {
    entries.forEach((entry) => {
      if (entry.isIntersecting) {
        const { target } = entry;
        const observer = this.getObserverByInstance(observerInstance);
        const observable = this.findObservableInObserver(observer, target);
        if (observer && observable) {
          sendIntersectionTimeMark(observer.name, observable.localMeta).catch((_error) => {});
        }
      }
    });
  };

  public addObservable = (observerName: string, target: Target, localMeta = {}) => {
    const observer = this.getObserverByName(observerName, true);
    if (this.targetIsObservable(target)) {
      return observer;
    }
    const observable = {
      localMeta,
      target,
    };
    observer.observableCollection.push(observable);
    observer.instance.observe(target);
    return observer;
  };

  public removeObserver = (observerName: string) => {
    const observer = this.observerCollection[observerName];
    if (observer) {
      if (observer.observableCollection.length === 0) {
        delete this.observerCollection[observerName];
        return;
      }
      for (let i = observer.observableCollection.length; i > 0; i--) {
        this.removeObservable(observer.name, observer.observableCollection[i - 1].target);
      }
    }
  };

  public removeObservable = (observerName: string, target: Target) => {
    const observer = this.getObserverByName(observerName);
    if (this.findObservableInObserver(observer, target)) {
      observer.instance.unobserve(target);
      observer.observableCollection = observer.observableCollection.filter(
        (observable) => observable.target !== target,
      );
      if (observer.observableCollection.length === 0) {
        this.removeObserver(observerName);
      }
    }
  };
}

export default new IntersectionWatcher();
