import { Template, TemplateByGroup, TemplateByType } from 'types/TemplateProvider';
import { TemplateApiDTO } from 'types/TemplateApiDTO';
import { MailTemplateGroup } from './MailTemplateProvider.types';

export class MailTemplateNormalizer {
  public static createGroupId(type: string, groupId: string) {
    return `${type}_${groupId}`;
  }

  public static formatTemplate(template: TemplateApiDTO): Template {
    return { ...template, id: String(template.id) };
  }

  public templateByType = new Map<string, TemplateByType>();
  public templateOrGroupById = new Map<string, Template | TemplateByGroup>();

  constructor(private templates: MailTemplateGroup[]) {
    this.parseTemplate = this.parseTemplate.bind(this);

    this.parse();
  }

  parse() {
    this.templates.forEach(this.parseTemplate);
  }

  private parseTemplate(templateGroup: MailTemplateGroup) {
    const templateGroupByType: TemplateByType = {
      id: templateGroup.id,
      caption: templateGroup.caption,
      itemIds: [],
    };

    this.templateByType.set(templateGroup.id, templateGroupByType);

    templateGroup.items.forEach((template) => {
      const formattedTemplate = MailTemplateNormalizer.formatTemplate(template);
      formattedTemplate.typeId = templateGroup.id;
      this.templateOrGroupById.set(formattedTemplate.id, formattedTemplate);

      if (template.type) {
        const group = this.getOrCreateTemplateByGroup(
          MailTemplateNormalizer.createGroupId(templateGroup.id, template.type),
          template.typeName,
        );

        if (!this.templateOrGroupById.has(group.id)) {
          templateGroupByType.itemIds.push(group.id);
          this.templateOrGroupById.set(group.id, group);
        }

        formattedTemplate.groupId = group.id;

        group.itemIds.push(String(template.id));
      } else {
        templateGroupByType.itemIds.push(formattedTemplate.id);
      }
    });
  }

  private getOrCreateTemplateByGroup(groupId, groupCaption): TemplateByGroup {
    if (!this.templateOrGroupById.has(groupId)) {
      return {
        id: groupId,
        caption: groupCaption,
        itemIds: [],
      };
    }

    return this.templateOrGroupById.get(groupId) as TemplateByGroup;
  }
}
