import {
  TemplateProvider,
  TemplateItemAny,
  TemplateFilter,
  TemplateByType,
  Template,
} from 'types/TemplateProvider';
import { MailTemplateGroup } from './MailTemplateProvider.types';
import { MailTemplateNormalizer } from './MailTemplateNormalizer';

export class MailTemplateProvider implements TemplateProvider {
  private mailTemplateNormalizer: MailTemplateNormalizer;

  constructor(templates: MailTemplateGroup[]) {
    this.mailTemplateNormalizer = new MailTemplateNormalizer(templates);
  }

  getTemplate(id: string): Template {
    if (!this.hasTemplate(id)) {
      throw new Error(`Has no template with id: ${id}`);
    }

    return this.mailTemplateNormalizer.templateOrGroupById.get(id) as Template;
  }

  getTemplateListByFilter(filter: TemplateFilter): TemplateItemAny[] {
    const typeGroup = this.mailTemplateNormalizer.templateByType.get(filter.typeId);
    if (!typeGroup) {
      return [];
    }

    const isTemplateSatisfyFilter = (template: Template): boolean => {
      if (!filter.text) {
        return true;
      }

      const filterTextLowerCase = filter.text.toLowerCase();
      return Boolean(
        template.bodyHtml.toLowerCase().indexOf(filterTextLowerCase) !== -1 ||
          (template.name && template.name.toLowerCase().indexOf(filterTextLowerCase) !== -1),
      );
    };

    return typeGroup.itemIds
      .map((id) => {
        const item = this.mailTemplateNormalizer.templateOrGroupById.get(id)!;

        if ('itemIds' in item) {
          const group = {
            ...item,
            items: item.itemIds
              .map((id) => {
                const template = this.mailTemplateNormalizer.templateOrGroupById.get(
                  id,
                ) as Template;

                if (isTemplateSatisfyFilter(template)) {
                  return template;
                }
              })
              .filter(Boolean),
          };

          if (!group.items.length) {
            return null;
          }

          return group;
        }

        if (isTemplateSatisfyFilter(item)) {
          return item;
        }

        return null;
      })
      .filter(Boolean) as TemplateItemAny[];
  }

  hasTemplate(id: string): boolean {
    return this.mailTemplateNormalizer.templateOrGroupById.has(id);
  }

  getTypes(): TemplateByType[] {
    return Array.from(this.mailTemplateNormalizer.templateByType.values());
  }
}
