import { makeObservable, observable, action } from 'mobx';
import { ListStore } from 'types/ListStore';

export class NormalizedListStore<Item> implements ListStore<Item> {
  private itemsMap = new Map<string, Item>();
  private orderIds: string[] = [];

  constructor() {
    makeObservable<ListStore<Item>, 'itemsMap' | 'orderIds'>(this, {
      addItemById: action,
      clear: action,
      itemsMap: observable,
      orderIds: observable,
    });
  }

  addItemById(id: string, item: Item): boolean {
    if (this.hasItemById(id)) {
      return false;
    }

    this.itemsMap.set(id, item);
    this.orderIds.push(id);

    return true;
  }

  clear() {
    this.itemsMap.clear();
    this.orderIds = [];
  }

  getItemById(id: string): Item | undefined {
    return this.itemsMap.get(id);
  }

  getOrderIds(): string[] {
    return this.orderIds;
  }

  removeItemById(id: string): boolean {
    if (!this.hasItemById(id)) {
      return false;
    }

    this.itemsMap.delete(id);
    this.orderIds = this.orderIds.filter((orderId) => orderId !== id);

    return true;
  }

  setItemById(id: string, item: Item) {
    if (!this.hasItemById(id)) {
      this.addItemById(id, item);
    } else {
      this.itemsMap.set(id, item);
    }
  }

  hasItemById(id: string) {
    return this.itemsMap.has(id);
  }
}
