import Bluebird from 'bluebird';
import { makeObservable, observable, runInAction } from 'mobx';
import { PageableItemDTO } from 'types/pagination/PageableItemDTO';
import { PageableDTO } from 'types/pagination/PageableDTO';
import { PaginationLoader } from 'types/PaginationLoader';
import { PaginationLoadCallback } from 'types/PaginationLoadCallback';
import { Loadable } from 'types/Loadable';
import { Destroyable } from 'types/Destroyable';
import { LoadMeta } from 'types/LoadMeta';
import { Loader } from '../Loader';
import { LoadMetaImpl } from '../LoadMeta';

export class PaginationLoaderByCallback<
  Item extends PageableItemDTO,
  Data extends PageableDTO<Item>
> implements PaginationLoader<Data>, Destroyable {
  private nextLoadUrl?: string;

  private loader: Loadable<Data> & Destroyable;
  public readonly meta: LoadMeta<Data>;

  public hasNextPage: boolean;
  public isFirstPageLoaded: boolean = false;

  constructor(
    private onLoad: PaginationLoadCallback<Item, Data>,
    private onLoadItems: (items: Item[]) => void,
  ) {
    this.handleLoad = this.handleLoad.bind(this);
    this.load = this.load.bind(this);

    this.meta = new LoadMetaImpl();
    this.loader = new Loader(this.meta, this.handleLoad);

    makeObservable(this, {
      hasNextPage: observable,
      isFirstPageLoaded: observable,
    });
  }

  public load() {
    return this.loader.load();
  }

  public destroy() {
    this.loader.destroy();
  }

  private handleLoad() {
    if (this.isFirstPageLoaded && !this.nextLoadUrl) {
      return Bluebird.reject(new Error('load data has no next url'));
    }

    return this.onLoad(this.nextLoadUrl).then((response) => {
      runInAction(() => {
        this.isFirstPageLoaded = true;
        this.nextLoadUrl = response.pagination.next;
        this.hasNextPage = Boolean(response.pagination.next);
        this.onLoadItems(response.data);
      });

      return response;
    });
  }
}
