import { EventTarget, defineEventAttribute } from 'event-target-shim';
import {
  PerformanceMetrikaEventAttributes,
  PerformanceMetrikaEvents,
  PerformanceMetrikaEventType,
} from './PerformanceMetrika.types';
import { PerformanceMetrikaBufferError } from './PerformanceMetrikaBufferError';

export class PerformanceMetrika extends EventTarget<
  PerformanceMetrikaEvents,
  PerformanceMetrikaEventAttributes
> {
  public static hasNeededFeatures(): boolean {
    return (
      'setResourceTimingBufferSize' in window.performance &&
      'onresourcetimingbufferfull' in window.performance &&
      'getEntriesByType' in window.performance &&
      'clearResourceTimings' in window.performance
    );
  }

  public static filterYandexTeamEvents(items: PerformanceEntryList) {
    return items.filter((item) => /yandex-team\.ru/.test(item.name));
  }

  public static MAX_BUFFER_SIZE = 20;

  public run() {
    if (PerformanceMetrika.hasNeededFeatures()) {
      this.subscribeToEvents();
    }
  }

  public stop() {
    this.unsubscribeFromEvents();
  }

  public destroy() {
    this.unsubscribeFromEvents();
  }

  private unsubscribeFromEvents() {
    performance.removeEventListener('resourcetimingbufferfull', this.handleBufferFull);
  }

  private subscribeToEvents() {
    performance.setResourceTimingBufferSize(PerformanceMetrika.MAX_BUFFER_SIZE);
    performance.addEventListener('resourcetimingbufferfull', this.handleBufferFull);
  }

  private handleBufferFull = () => {
    this.sendFilteredData();
    performance.clearResourceTimings();
    performance.setResourceTimingBufferSize(PerformanceMetrika.MAX_BUFFER_SIZE);
  };

  private sendFilteredData() {
    const items = PerformanceMetrika.filterYandexTeamEvents(
      performance.getEntriesByType('resource'),
    );

    if (!items.length) {
      return;
    }

    if (items.length > PerformanceMetrika.MAX_BUFFER_SIZE) {
      this.dispatchEvent(
        new CustomEvent(PerformanceMetrikaEventType.Error, {
          detail: new PerformanceMetrikaBufferError(items.length),
        }),
      );
      return;
    }

    this.dispatchEvent(new CustomEvent(PerformanceMetrikaEventType.BufferFull, { detail: items }));
  }
}

defineEventAttribute(PerformanceMetrika.prototype, 'bufferfull');
defineEventAttribute(PerformanceMetrika.prototype, 'error');
