import { BackendError } from 'api/BackendError';
import { ErrorLevelTypes } from 'services/Logger/Logger.types';
import flatten from 'flat';
import { Config } from 'types/Config';
import RumProvider from '../RumProvider';
import { getSearchParam, getMessageFromError, getCurrentPage } from './utils';

const isNeedSend = process.env.NODE_ENV === 'production' || process.env.NODE_ENV === 'test';

export class RumError {
  public rumInstance;
  private readonly instance: RumError;

  constructor() {
    if (this.instance) {
      return this.instance;
    }
    try {
      this.rumInstance = RumProvider.rumInstance;
      this.instance = this;
    } catch (error) {
      throw Error(error);
    }
  }

  private getCommonAdditional = () => {
    const rumAdditional = RumProvider.getGlobalAdditional() as Partial<Config>;
    return {
      page: location.href,
      user: { id: rumAdditional?.user?.id, login: rumAdditional?.user?.login },
    };
  };

  public logBackendError = (backendError: BackendError) => {
    if (!isNeedSend) {
      return;
    }
    this.rumInstance?.logError({
      message: getMessageFromError(backendError),
      source: 'backend',
      page: getCurrentPage(),
      sourceMethod: backendError.status,
      type: backendError.status === 0 ? 'network' : backendError.status,
      level: this.rumInstance.ERROR_LEVEL.ERROR,
      additional: flatten({
        timelineItemName: getMessageFromError(backendError),
        ...backendError.additional,
        wid: getSearchParam('_wid', backendError.additional?.url),
        errorCode: backendError.status,
        ...this.getCommonAdditional(),
      }),
      ignoreErrorMessage: true,
    });
  };

  public logFrontendError = (
    error: Error,
    errorLevel?: ErrorLevelTypes,
    localAdditional?: unknown,
  ) => {
    if (!isNeedSend) {
      return;
    }
    const message = `FrontendError: ${error.message}`;
    const level = errorLevel
      ? this.rumInstance.ERROR_LEVEL[errorLevel]
      : this.rumInstance.ERROR_LEVEL.ERROR;
    this.rumInstance?.logError(
      {
        message,
        source: 'frontend',
        page: getCurrentPage(),
        type: 'logic',
        level,
        additional: flatten({
          localAdditional,
          timelineItemName: message,
          ...this.getCommonAdditional(),
        }),
        ignoreErrorMessage: true,
      },
      error,
    );
  };
}

export default new RumError();
