import flatten from 'flat';
import { history } from 'entry/appHistory';
import { Config } from 'types/Config';
import { AdditionalType } from './RumProvider.types';
import Rum from './Rum';

export const isObject = (variable) => {
  return Object.prototype.toString.call(variable) === '[object Object]';
};

export class RumProvider {
  public rumInstance;
  private readonly instance: RumProvider;
  public globalAdditional: AdditionalType = {};

  constructor() {
    if (this.instance) {
      return this.instance;
    }
    try {
      this.rumInstance = Rum;
      this.instance = this;
      this.listenHistory();
    } catch (error) {
      throw Error(error);
    }
  }

  private listenHistory = (): void => {
    history.listen((location) => {
      const currentUrl = window.location.href;
      const newPage = location.pathname.split('/')[1];
      const currentPage = this.getGlobalParam('-page');
      if (newPage !== currentPage) {
        this.setGlobalParam('-page', newPage);
      }
      this.setGlobalParam('url', currentUrl);
      this.setGlobalAdditional({ url: currentUrl });
    });
  };

  public sendTimeMark = (
    counterId: string,
    time?: number,
    additional: AdditionalType = {},
    withGlobal: boolean = true,
  ): void => {
    if (withGlobal) {
      additional = { global: this.globalAdditional, ...additional };
    }
    const payload = { '-additional': this.prepareAdditional(flatten(additional)) };
    this.rumInstance.sendTimeMark(counterId, time, undefined, payload);
  };

  public makeSubPage = (subpageName: string) => {
    return this.rumInstance.makeSubPage(subpageName);
  };

  public prepareAdditional = (additional: AdditionalType): string => {
    return encodeURIComponent(JSON.stringify(additional));
  };

  public setGlobalParam = (name: string, value: unknown) => {
    this.rumInstance.setVars({ [name]: value });
  };

  public getGlobalAdditional = (): AdditionalType => {
    return this.globalAdditional;
  };

  public setGlobalAdditional = (additional: AdditionalType = {}): void => {
    if (isObject(additional)) {
      this.globalAdditional = { ...this.globalAdditional, ...additional };
      return;
    }
    throw Error('incorrect type of additional');
  };

  public setGlobalParams = (params: AdditionalType = {}): void => {
    if (isObject(params)) {
      this.rumInstance.setVars(params);
      return;
    }
    throw Error('incorrect type of params');
  };

  public getGlobalParam = (name: string): unknown | undefined => {
    return this.rumInstance._vars[name];
  };

  public setUpdateFromInfo = (infoData: Config): void => {
    const rawAdditional = { user: infoData.user, features: infoData.features };
    this.setGlobalAdditional(rawAdditional);
    this.rumInstance?.updateErrors?.({ version: window.__APP_VERSION__ });
    this.setGlobalParams({
      '-env': infoData.testMode ? 'testing' : 'production',
      '-version': window.__APP_VERSION__,
    });
  };
}

export default new RumProvider();
