import { makeAutoObservable } from 'mobx';
import Bluebird from 'bluebird';
import { get } from 'api/common';
import { UserStatusesResponseDTO, UserDTO } from 'types/UserStatusesResponseDTO';
import { UserStatus } from './UserStatus';
import { UserStatusStore } from './UserStatusStore';
import { UserStatusConstructor } from './UserStatusConstructor';

export class UserStatusStoreImpl implements UserStatusStore {
  statuses = new Map<number, UserStatus>();

  constructor(private readonly UserStatusConstructorImpl: UserStatusConstructor) {
    this.UserStatusConstructorImpl = UserStatusConstructorImpl;
    makeAutoObservable<UserStatusStore, 'UserStatusConstructorImpl'>(this, {
      UserStatusConstructorImpl: false,
    });
  }

  getUserStatusById(id: number) {
    let status = this.statuses.get(id);

    if (status) {
      return status;
    }

    status = new this.UserStatusConstructorImpl(this, id);
    this.statuses.set(id, status);

    return status;
  }

  removeUserStatusById(id: number) {
    this.statuses.delete(id);
  }

  syncStatuses() {
    if (!this.statuses.size) {
      return Bluebird.resolve([]);
    }

    return get<UserStatusesResponseDTO>({
      url: '/users/status',
      data: { ids: Array.from(this.statuses.keys()) },
      global: false,
    }).then((data) => {
      this.updateStatuses(data);
      return data;
    });
  }

  private updateStatuses = (data: UserStatusesResponseDTO) => {
    data.users?.forEach(this.updateStatus);
  };

  private updateStatus = (data: UserDTO) => {
    const status = this.statuses.get(data.id);

    if (!status) {
      return;
    }

    if (status.status.id === data.status.id) {
      return;
    }

    status.status = data.status;
  };
}
