import {
  Subject,
  Subscription,
  combineLatest,
  from,
  EMPTY,
  of,
  Observable,
  interval,
  defer,
  throwError,
  merge,
} from 'rxjs';
import {
  tap,
  switchMap,
  catchError,
  retryWhen,
  takeUntil,
  filter,
  startWith,
  mapTo,
} from 'rxjs/operators';
import { IndexLocationWithAlign } from 'react-virtuoso';
import { makeObservable, action, computed, runInAction } from 'mobx';
import { BackendError } from 'api/BackendError';
import { LinkedListImpl } from 'utils/LinkedList';
import { createObservableFromMobxReaction } from 'utils/createObservableFromMobxReaction';
import {
  increasingTimerRetryStrategy,
  IncreasingTimerRetryStrategyOptions,
} from 'utils/increasingTimerRetryStrategy';
import {
  VirtualListMeta as IVirtualListMeta,
  VirtualList,
  VirtualListItem,
  VirtualListItemId,
  VirtualListDataProvider,
  VirtualListGetDirection,
  VirtualListGetOptions,
  VirtualListService as IVirtualListService,
  VirtualListServiceLoadInitOptions,
} from 'types/VirtualList';
import { logger } from 'services/Logger';
import { VirtualListMeta } from './VirtualListMeta';
import { AsyncTask } from './AsyncTask';

export class VirtualListService<Item extends VirtualListItem = VirtualListItem>
  implements IVirtualListService<Item> {
  public static readonly PAGE_ITEMS_LIMIT = 25;

  public static readonly META_POLLING_INTERVAL_MS = 60000;

  private static readonly INCREASE_TIMER_RETRY_STRATEGY_OPTIONS: IncreasingTimerRetryStrategyOptions = {
    scalingDuration: 5000,
  };

  private startReached$ = new Subject<VirtualListItemId | undefined>();
  private endReached$ = new Subject<VirtualListItemId | undefined>();
  private subscription: Subscription = new Subscription();

  private init$ = new Subject<VirtualListServiceLoadInitOptions<Item> | undefined>();
  private loadNext$ = new Subject<void>();
  private loadPrevious$ = new Subject<void>();

  private initOptions: VirtualListServiceLoadInitOptions;

  private initialTopMostItemId: VirtualListItemId | null = null;

  private listMeta = new VirtualListMeta();
  private store = new Map<VirtualListItemId, Item>();
  private linkedListOfIds = new LinkedListImpl<VirtualListItemId>();

  loadInitTask = new AsyncTask();
  loadNextTask = new AsyncTask();
  loadPreviousTask = new AsyncTask();

  constructor(private dataProvider: VirtualListDataProvider<Item>) {
    this.handleInitData = this.handleInitData.bind(this);
    this.createWatchers = this.createWatchers.bind(this);

    makeObservable<
      VirtualListService<Item>,
      | 'listIds'
      | 'appendNonOrderItems'
      | 'hasMorePrevious'
      | 'hasMoreNext'
      | 'updateMeta'
      | 'startReached'
      | 'endReached'
      | 'tryAddItemsToEnd'
      | 'tryAddItemsToStart'
      | 'retryInit'
      | 'retryLoadNext'
      | 'retryLoadPrevious'
    >(this, {
      appendNonOrderItems: action,
      updateMeta: action,
      listIds: computed,
      firstItemIndex: computed,
      hasMorePrevious: computed,
      hasMoreNext: computed,
      startReached: action.bound,
      endReached: action.bound,
      tryAddItemsToEnd: action.bound,
      tryAddItemsToStart: action.bound,
      retryInit: action.bound,
      retryLoadNext: action.bound,
      retryLoadPrevious: action.bound,
    });

    this.subscription.add(this.initPageLoad().subscribe());
  }

  get firstItemIndex() {
    const item = this.store.get(this.linkedListOfIds.first?.value ?? '');
    return item?.seqId ?? 0;
  }

  get hasMorePrevious() {
    return (
      Boolean(this.listMeta.firstId) && this.listMeta.firstId !== this.linkedListOfIds.first?.value
    );
  }

  get hasMoreNext() {
    return (
      Boolean(this.listMeta.lastId) && this.listMeta.lastId !== this.linkedListOfIds.last?.value
    );
  }

  get initialTopMostItemIndex(): number | IndexLocationWithAlign {
    if (!this.initOptions) {
      return 0;
    }

    if (this.initialTopMostItemId) {
      const indexForTopMostItemId = this.listIds.indexOf(this.initialTopMostItemId);

      if (!indexForTopMostItemId) {
        return 0;
      }

      return indexForTopMostItemId;
    }

    if (this.initOptions.from === 'end') {
      return { align: 'end', index: 'LAST' };
    }

    return 0;
  }

  get length(): number {
    return this.linkedListOfIds.count;
  }

  init(options: VirtualListServiceLoadInitOptions<Item>) {
    this.initOptions = options;
    this.init$.next();
  }

  retryInit() {
    this.init$.next();
  }

  retryLoadNext() {
    this.loadNext$.next();
  }

  retryLoadPrevious() {
    this.loadPrevious$.next();
  }

  destroy() {
    this.subscription.unsubscribe();
  }

  hasItemByAbsoluteIndex(index: number): boolean {
    return this.store.has(this.listIds[index - this.firstItemIndex]);
  }

  getItemIdByAbsoluteIndex(index: number): VirtualListItemId {
    const item = this.getItemByAbsoluteIndex(index);
    return item.id;
  }

  getItemByAbsoluteIndex(index: number): Item {
    if (!this.hasItemByAbsoluteIndex(index)) {
      throw new Error(`Has no item data for absolute index: ${index}`);
    }
    return this.store.get(this.listIds[index - this.firstItemIndex])!;
  }

  appendNonOrderItems(items: Item[]) {
    const sortedItemsByOrder = items.slice().sort((a, b) => a.seqId - b.seqId);
    this.tryAddItemsToEnd(sortedItemsByOrder);
    this.tryAddItemsToStart(sortedItemsByOrder);
  }

  updateMeta(meta: IVirtualListMeta) {
    this.listMeta.updateFromJson(meta);
  }

  startReached() {
    this.startReached$.next(this.linkedListOfIds.first?.value);
  }

  endReached() {
    this.endReached$.next(this.linkedListOfIds.last?.value);
  }

  private initPageLoad() {
    return this.init$.pipe(
      switchMap(() => this.createInitLoader().pipe(switchMap(() => this.createWatchers()))),
    );
  }

  private createInitLoader() {
    const loadOptions: VirtualListGetOptions = {
      limit: VirtualListService.PAGE_ITEMS_LIMIT,
    };

    if (this.initOptions.from === 'end') {
      loadOptions.direction = 'previous';
    }

    if (this.initOptions.from === 'start') {
      loadOptions.direction = 'next';
    }

    if (this.initOptions.from === 'center') {
      loadOptions.exactId = this.initOptions.exactId;
    }

    this.loadInitTask.run();

    return defer(() => from(this.dataProvider.get(loadOptions))).pipe(
      catchError((error) => {
        if (error instanceof BackendError && [400, 404].includes(error.status)) {
          delete loadOptions.exactId;
          loadOptions.direction = 'previous';
          this.initOptions.from = 'end';
        }

        return throwError(error);
      }),
      retryWhen(
        increasingTimerRetryStrategy(VirtualListService.INCREASE_TIMER_RETRY_STRATEGY_OPTIONS),
      ),
      tap((data) => {
        this.handleInitData(data);
        this.loadInitTask.done();
      }),
      catchError((error) => {
        logger.reportAppErrorOnly(error);
        this.loadInitTask.reject(error);

        return EMPTY;
      }),
    );
  }

  private handleInitData(data: VirtualList<Item>) {
    this.calculateInitialTopMostItemId(data.data);

    this.listMeta.updateFromJson(data.meta);

    if (this.initOptions.from === 'start') {
      this.tryAddItemsToStart(data.data);
    } else {
      this.tryAddItemsToEnd(data.data);
    }
  }

  private calculateInitialTopMostItemId(items: Item[]) {
    if (this.initOptions.from !== 'center') {
      return;
    }

    this.initialTopMostItemId = this.initOptions.getInitialTopMostItemId(items);
  }

  private createWatchers() {
    return merge(
      this.createWatchAutoLoadPrev(),
      this.createWatchAutoLoadNext(),
      this.createWatchMeta(),
      this.createWatchInitDataNeedToLoad(),
    );
  }

  private createWatchAutoLoadPrev() {
    return this.createWatcherAutoLoad({
      reached$: this.startReached$,
      getEdgeId: () => this.linkedListOfIds.first?.value,
      getHasMore: () => this.hasMorePrevious,
      direction: 'previous',
      handleAddItems: this.tryAddItemsToStart,
      asyncTask: this.loadPreviousTask,
      retry$: this.loadPrevious$,
    });
  }

  private createWatchAutoLoadNext() {
    return this.createWatcherAutoLoad({
      reached$: this.endReached$,
      getEdgeId: () => this.linkedListOfIds.last?.value,
      getHasMore: () => this.hasMoreNext,
      direction: 'next',
      handleAddItems: this.tryAddItemsToEnd,
      asyncTask: this.loadNextTask,
      retry$: this.loadNext$,
    });
  }

  private createWatcherAutoLoad({
    reached$,
    getEdgeId,
    getHasMore,
    direction,
    handleAddItems,
    asyncTask,
    retry$,
  }: {
    reached$: Observable<VirtualListItemId | undefined>;
    getEdgeId: () => VirtualListItemId | undefined;
    getHasMore: () => boolean;
    direction: VirtualListGetDirection;
    handleAddItems: (items: Item[]) => void;
    asyncTask: AsyncTask;
    retry$: Subject<void>;
  }) {
    asyncTask.idle();

    const hasMore$ = createObservableFromMobxReaction(getHasMore, {
      fireImmediately: true,
    });

    const edgeId$ = createObservableFromMobxReaction(getEdgeId, {
      fireImmediately: true,
    });

    return combineLatest([reached$, edgeId$]).pipe(
      switchMap(([reached, edgeId]) => {
        if (edgeId !== undefined && reached === edgeId) {
          return combineLatest([hasMore$, of(edgeId)]);
        }

        return EMPTY;
      }),
      switchMap((value) => retry$.pipe(startWith(undefined), mapTo(value))),
      switchMap(([hasMore, edgeId]) => {
        if (!hasMore) {
          return EMPTY;
        }

        asyncTask.run();

        return defer(() =>
          from(
            this.dataProvider.get({
              fromId: edgeId,
              excludeFrom: Boolean(edgeId),
              limit: VirtualListService.PAGE_ITEMS_LIMIT,
              direction,
            }),
          ),
        ).pipe(
          retryWhen(
            increasingTimerRetryStrategy(VirtualListService.INCREASE_TIMER_RETRY_STRATEGY_OPTIONS),
          ),
          catchError((error) => {
            logger.reportAppErrorOnly(error);
            asyncTask.reject(error);
            return EMPTY;
          }),
        );
      }),
      tap((data) => {
        runInAction(() => {
          asyncTask.done();
          handleAddItems(data.data);
          this.updateMeta(data.meta);
        });
      }),
    );
  }

  private createWatchInitDataNeedToLoad() {
    const length$ = createObservableFromMobxReaction(() => this.length, { fireImmediately: true });
    const metaVersion$ = createObservableFromMobxReaction(() => this.listMeta.lastSeqId, {
      fireImmediately: true,
    });

    return metaVersion$.pipe(
      filter((metaVersion) => metaVersion > 0),
      takeUntil(length$.pipe(filter((length) => length > 0))),
      switchMap(() => this.createInitLoader()),
    );
  }

  private createWatchMeta() {
    return interval(VirtualListService.META_POLLING_INTERVAL_MS).pipe(
      switchMap(() => {
        return from(this.dataProvider.getMeta()).pipe(catchError(() => EMPTY));
      }),
      tap((meta) => this.updateMeta(meta)),
    );
  }

  private tryAddItemsToEnd(sortedItemsByOrder: Item[]) {
    sortedItemsByOrder.forEach((item) => {
      if (!this.linkedListOfIds.last || this.linkedListOfIds.last?.value === item.prevId) {
        this.linkedListOfIds.addLast(item.id);
        this.store.set(item.id, item);
      }
    });
  }

  private tryAddItemsToStart(sortedItemsByOrder: Item[]) {
    sortedItemsByOrder
      .slice()
      .reverse()
      .forEach((item) => {
        const firstId = this.linkedListOfIds.first?.value;

        const firstItem = firstId && this.store.get(firstId);

        if (!firstItem || firstItem.prevId === item.id) {
          this.linkedListOfIds.addFirst(item.id);
          this.store.set(item.id, item);
        }
      });
  }

  private get listIds(): VirtualListItemId[] {
    return this.linkedListOfIds.asArray;
  }
}
