import { Observable, BehaviorSubject, EMPTY, merge, timer } from 'rxjs';
import { switchMap, switchMapTo, tap, filter, map } from 'rxjs/operators';
import { XivaMessageAny } from 'types/xiva/XivaWebsocketJSONApi';
import { UniqIdGenerator } from 'utils/UniqIdGenerator';
import { isPong } from './xivaMessageTypeSafeCheck';

export interface CreateXivaMultiplexOptions {
  messageFromServer: Observable<XivaMessageAny>;
  messageToServer: (next: XivaMessageAny) => void;
  isWebsocketOpened: BehaviorSubject<boolean>;
}

export class XivaAliveCheck {
  static readonly START_PING_DELAY_MS = 15000;
  static readonly TIME_BETWEEN_PINGS_MS = 15000;

  static create(options: CreateXivaMultiplexOptions) {
    return new XivaAliveCheck(options).observable;
  }

  private readonly _observable: Observable<never>;

  private lastPingRequestId = '';
  private lastPongRequestId = '';

  constructor(private options: CreateXivaMultiplexOptions) {
    this._observable = this.connectIfWebsocketOpen();
  }

  get observable() {
    return this._observable;
  }

  private connectIfWebsocketOpen() {
    return this.options.isWebsocketOpened.pipe(
      switchMap((isOpen) => (isOpen ? this.startPingPong() : EMPTY)),
    );
  }

  private startPingPong() {
    return merge(this.listenForPong(), this.sendPing()).pipe(switchMapTo(EMPTY));
  }

  private sendPing() {
    return timer(XivaAliveCheck.START_PING_DELAY_MS, XivaAliveCheck.TIME_BETWEEN_PINGS_MS).pipe(
      tap(() => {
        if (!this.lastPingRequestId) {
          return;
        }

        if (this.lastPingRequestId === this.lastPongRequestId) {
          return;
        }

        throw new Error('XIVA_NO_PONG');
      }),
      map(() => UniqIdGenerator.global.next()),
      tap((requestId) => {
        this.lastPingRequestId = requestId;
        this.options.messageToServer({ method: '/ping', params: {}, id: requestId });
      }),
    );
  }

  private listenForPong() {
    return this.options.messageFromServer.pipe(
      filter(isPong),
      filter((pong) => !pong.error),
      tap((pong) => {
        this.lastPongRequestId = pong.id ?? '';
      }),
    );
  }
}
