import { XivaSubscriptionParamsAny, XivaMessageAny } from 'types/xiva/XivaWebsocketJSONApi';
import { webSocket } from 'rxjs/webSocket';
import { timer, Observable, BehaviorSubject, merge, defer } from 'rxjs';
import { retryWhen, delayWhen, share, finalize, tap } from 'rxjs/operators';
import { logger } from 'services/Logger';
import { XivaMultiplexOptions } from 'types/xiva/XivaMultiplexOptions';
import { XivaClient as IXivaClient } from 'types/xiva/XivaClient';
import { XivaClientLog } from './XivaClientLog';
import { XivaMultiplex } from './XivaMultiplex';
import { XivaAliveCheck } from './XivaAliveCheck';
import { getMultiplexHash } from './getMultiplexHash';
import { XivaClientCloseLog } from './XivaClientCloseLog';
import { XivaClientOpenLog } from './XivaClientOpenLog';

export interface XivaOptions {
  getXivaSettings?: (options?: XivaMultiplexOptions) => Promise<XivaSubscriptionParamsAny>;
  url?: string;
}

interface XivaOptionsInner {
  getXivaSettings: (options?: XivaMultiplexOptions) => Promise<XivaSubscriptionParamsAny>;
  url: string;
}

export class XivaClient<T extends XivaMessageAny> implements IXivaClient<T> {
  static readonly DEFAULT_OPTIONS: XivaOptionsInner = {
    url: 'wss://push.yandex-team.ru/websocketapi',
    getXivaSettings: () => Promise.resolve({ uid: '', service: '', client: '', session: '' }),
  };

  private multiplexCache = new Map<string, Observable<T>>();

  private readonly options: XivaOptionsInner;

  public messageFromServer: Observable<T>;

  public isWebsocketOpened = new BehaviorSubject<boolean>(false);

  public next: (value: T) => void = (message) => {
    this.innerNext(message);
  };

  private innerNext: (value: T) => void = () => {};

  private websocketInstanceId = 0;

  constructor(options?: XivaOptions) {
    this.options = { ...XivaClient.DEFAULT_OPTIONS, ...options };

    this.createWebsocket();
  }

  multiplex(options: XivaMultiplexOptions = {}) {
    const multiplexHash = getMultiplexHash(options);

    if (this.multiplexCache.get(multiplexHash)) {
      return this.multiplexCache.get(multiplexHash)!;
    }

    const observable = XivaMultiplex.create({
      messageFromServer: this.messageFromServer,
      messageToServer: this.next,
      isWebsocketOpened: this.isWebsocketOpened,
      getXivaCredits: () => this.options.getXivaSettings(options),
    }).pipe(
      finalize(() => {
        this.multiplexCache.delete(multiplexHash);
      }),
      share(),
    ) as Observable<T>;

    this.multiplexCache.set(multiplexHash, observable);

    return observable;
  }

  private createWebsocket = () => {
    this.messageFromServer = defer(() => {
      this.websocketInstanceId += 1;

      const currentWebsocketInstanceId = this.websocketInstanceId;

      const webSocketSubject = webSocket<T>({
        url: this.options.url,
        openObserver: {
          next: () => {
            logger.reportInfo(
              new XivaClientOpenLog(undefined, {
                websocketInstanceId: currentWebsocketInstanceId,
              }),
            );
            this.isWebsocketOpened.next(true);
          },
        },
        closeObserver: {
          next: (closeEvent) => {
            logger.reportInfo(
              new XivaClientCloseLog(undefined, {
                code: closeEvent.code,
                reason: closeEvent.reason,
                wasClean: closeEvent.wasClean,
                websocketInstanceId: currentWebsocketInstanceId,
              }),
            );

            if (this.websocketInstanceId === currentWebsocketInstanceId) {
              this.isWebsocketOpened.next(false);
            }
          },
        },
      });

      this.innerNext = (message) => {
        logger.reportInfo(new XivaClientLog('to server', message));
        webSocketSubject.next(message);
      };

      return merge(
        webSocketSubject,
        XivaAliveCheck.create({
          isWebsocketOpened: this.isWebsocketOpened,
          messageFromServer: webSocketSubject,
          messageToServer: this.next,
        }),
      );
    }).pipe(
      tap((message) => {
        logger.reportInfo(new XivaClientLog('from server', message));
      }),
      retryWhen((errors) =>
        errors.pipe(
          tap((error) => {
            this.isWebsocketOpened.next(false);
            let formattedError = error;
            if (!(formattedError instanceof Error)) {
              formattedError = new Error('XIVA_CONNECTION_HOST_ERROR');
            }

            logger.reportError(formattedError);
          }),
          delayWhen(() => timer(1000)),
        ),
      ),
      share(),
    );
  };
}
