import { Observable, BehaviorSubject, merge, EMPTY, defer, timer } from 'rxjs';
import { XivaSubscriptionParamsAny, XivaMessageAny } from 'types/xiva/XivaWebsocketJSONApi';
import {
  retryWhen,
  delayWhen,
  share,
  switchMap,
  switchMapTo,
  tap,
  filter,
  finalize,
} from 'rxjs/operators';
import { UniqIdGenerator } from 'utils/UniqIdGenerator';
import {
  isXivaDisconnectedMessage,
  isXivaNotificationMessageAny,
  isXivaSubscribeMethodResponse,
  isXivaErrorMessage,
} from './xivaMessageTypeSafeCheck';

export interface CreateXivaMultiplexOptions {
  messageFromServer: Observable<XivaMessageAny>;
  messageToServer: (next: XivaMessageAny) => void;
  isWebsocketOpened: BehaviorSubject<boolean>;
  getXivaCredits: () => Promise<XivaSubscriptionParamsAny>;
}

export class XivaMultiplex {
  static RECONNECT_DELAY_MS = 5000;

  static create(options: CreateXivaMultiplexOptions) {
    return new XivaMultiplex(options).observable;
  }

  private readonly _observable: Observable<XivaMessageAny>;
  private subscriptionRequestId = '';
  private subscriptionToken = '';
  private xivaCredits?: XivaSubscriptionParamsAny;

  constructor(private options: CreateXivaMultiplexOptions) {
    this._observable = merge(this.connectIfWebsocketOpen(), this.listenMessages()).pipe(
      retryWhen((errors) => errors.pipe(delayWhen(() => timer(XivaMultiplex.RECONNECT_DELAY_MS)))),
      finalize(this.handleUnsubscribe),
      share(),
    );
  }

  get observable() {
    return this._observable;
  }

  private connectIfWebsocketOpen() {
    return this.options.isWebsocketOpened.pipe(
      switchMap((isOpen) => (isOpen ? this.createConnection() : EMPTY)),
    );
  }

  private createConnection() {
    return defer(this.options.getXivaCredits).pipe(
      tap(this.storeCredits),
      tap(this.sendConnectMessage),
      switchMapTo(EMPTY),
    );
  }

  private storeCredits = (credits: XivaSubscriptionParamsAny) => {
    this.xivaCredits = credits;
  };

  private sendConnectMessage = () => {
    if (!this.xivaCredits) {
      return;
    }

    this.subscriptionRequestId = UniqIdGenerator.global.next();
    this.options.messageToServer({
      method: '/subscribe',
      params: this.xivaCredits,
      id: this.subscriptionRequestId,
    });
  };

  private handleUnsubscribe = () => {
    if (!this.options.isWebsocketOpened.value || !this.xivaCredits || !this.subscriptionToken) {
      return;
    }

    const params = {
      ...this.xivaCredits,
      service: this.xivaCredits.service.split(':')[0],
      subscription_token: this.subscriptionToken,
    };

    this.options.messageToServer({
      method: '/unsubscribe',
      params,
    });

    this.cleanupConnectionData();
  };

  private listenMessages() {
    return this.options.messageFromServer.pipe(
      tap(this.handleConnected),
      filter(this.filterMultiplexMessage),
      tap(this.handleDisconnected),
      tap(this.handleXivaError),
    );
  }

  private handleConnected = (message: XivaMessageAny) => {
    if (!isXivaSubscribeMethodResponse(message)) {
      return;
    }

    if (message.id !== this.subscriptionRequestId) {
      return;
    }

    const token = message.result?.subscription_token;
    if (!token) {
      throw new Error('subscription error');
    }

    this.subscriptionToken = token;
  };

  private handleDisconnected = (message: XivaMessageAny) => {
    if (isXivaDisconnectedMessage(message)) {
      this.cleanupConnectionData();

      throw new Error('disconnected');
    }
  };

  private handleXivaError = (message: XivaMessageAny) => {
    if (isXivaErrorMessage(message)) {
      this.cleanupConnectionData();

      throw new Error('xivaws-error');
    }
  };

  private filterMultiplexMessage = (message: XivaMessageAny) => {
    if (!isXivaNotificationMessageAny(message)) {
      return false;
    }

    return message.params.subscription_token === this.subscriptionToken;
  };

  private cleanupConnectionData() {
    this.subscriptionRequestId = '';
    this.subscriptionToken = '';
    this.xivaCredits = undefined;
  }
}
