import React from 'react';
import PropTypes from 'prop-types';
import cx from 'classnames';
import Select from 'components/Select';
import TreeNode from '../TreeNode';
import defaultTheme from './styles.modules.scss';

export default class TreeView extends React.Component {
  getNode(id, level) {
    const RenderNode = this.renderNode;
    const Component = this.props.componentNode;

    const node = this.props.list[id];

    if (!node) {
      return null;
    }

    const label = Component ? (
      <Component {...node} />
    ) : (
      <Select value={node.name} range={node.searchRange} />
    );
    const selected = this.props.selectedMap && this.props.selectedMap[id];
    const onArrowClick = this.props.onArrowClick ? this.props.onArrowClick.bind(this, id) : null;
    const onItemClick =
      this.props.onItemClick && this.props.canItemClick(node)
        ? this.props.onItemClick.bind(this, id, node, selected)
        : onArrowClick;

    const collapsed =
      this.props.collapseMap && this.props.collapseMap[id] && !this.props.forceExpanded;

    return (
      <RenderNode
        labelClassName={this.props.labelClassName}
        key={id}
        level={level}
        node={node}
        nodeLabel={label}
        onClick={onItemClick}
        onArrowClick={onArrowClick}
        clickMode={this.props.clickMode}
        theme={this.props.theme}
        selected={selected}
        collapsed={collapsed}
        defaultCollapsed={this.props.defaultCollapsed}
      />
    );
  }

  renderNode = args => {
    const { level, node, ...props } = args;
    let children = null;

    if (node.items && node.items.length) {
      children = node.items.map(id => this.getNode(id, level + 1));
    }

    return (
      <TreeNode {...props} type={node.type} offset={level * this.props.offset}>
        {children}
      </TreeNode>
    );
  };

  render() {
    const { list, className, theme } = this.props;
    const level = 1;

    if (!list || !list.root) {
      return <div className={cx(theme.root_empty, className)} />;
    }

    return (
      <div className={cx(theme.root, className)}>
        <div className={cx(defaultTheme.root__wrap, theme.root__wrap)}>
          {list.root.items.map(id => this.getNode(id, level))}
        </div>
      </div>
    );
  }
}

TreeView.propTypes = {
  list: PropTypes.instanceOf(Object),
  labelClassName: PropTypes.string,
  componentNode: PropTypes.oneOfType([PropTypes.func, PropTypes.element]),
  showToolbar: PropTypes.bool,
  className: PropTypes.string,
  treeNodeTheme: PropTypes.instanceOf(Object),
  theme: PropTypes.instanceOf(Object),
  clickMode: PropTypes.string, // wrap, label
  selectedMap: PropTypes.instanceOf(Object),
  collapseMap: PropTypes.instanceOf(Object),
  defaultCollapsed: PropTypes.bool,
  forceExpanded: PropTypes.bool,
  offset: PropTypes.number,
  canItemClick: PropTypes.func,
  onArrowClick: PropTypes.func,
  onItemClick: PropTypes.func,
};

TreeView.defaultProps = {
  list: undefined,
  componentNode: undefined,
  className: undefined,
  labelClassName: undefined,
  onArrowClick: undefined,
  onItemClick: undefined,
  treeNodeTheme: undefined,
  selectedMap: undefined,
  collapseMap: undefined,
  defaultCollapsed: undefined,
  forceExpanded: undefined,
  showToolbar: true,
  clickMode: 'label',
  theme: defaultTheme,
  offset: 16,
  canItemClick: () => true,
};
