import {
  ArrayOperation,
  ArrayOperationRemoveItemData,
  ArrayOperationClearData,
  ArrayOperationAddItemData,
  ArrayOperationChangeItemData,
} from 'types/ArrayOperation';

export type ArrayOperationBuilderItemId = string | number;

export interface ArrayOperationBuilderItem {
  id: ArrayOperationBuilderItemId;
}

export class ArrayOperationBuilder<T extends ArrayOperationBuilderItem> {
  constructor(private items: T[]) {}

  removeItem(item: T): ArrayOperationRemoveItemData<T> | null {
    return this.removeItemById(item.id);
  }

  removeItemById(id: ArrayOperationBuilderItemId): ArrayOperationRemoveItemData<T> | null {
    const itemForRemove = this.getItemById(id);

    if (!itemForRemove) {
      return null;
    }

    const nextItems = this.items.filter((valueItem) => valueItem.id !== id);

    return {
      type: ArrayOperation.RemoveItem,
      array: { prev: this.items, next: nextItems },
      item: itemForRemove,
    };
  }

  addItem(item: T): ArrayOperationAddItemData<T> | null {
    const existedItem = this.getItemById(item.id);

    if (existedItem) {
      return null;
    }

    const changedItems = this.items.concat([item]);

    return {
      type: ArrayOperation.AddItem,
      array: { prev: this.items, next: changedItems },
      item,
    };
  }

  changeOrAddItem(updatedItem: T): ArrayOperationChangeItemData<T> | ArrayOperationAddItemData<T> {
    const index = this.getItemIndexById(updatedItem.id);
    if (index === -1) {
      return this.addItem(updatedItem)!;
    }

    const changedItems = [
      ...this.items.slice(0, index),
      updatedItem,
      ...this.items.slice(index + 1),
    ];
    const prevItem = this.items[index];
    return {
      type: ArrayOperation.ChangeItem,
      array: { prev: this.items, next: changedItems },
      item: { prev: prevItem, next: updatedItem },
    };
  }

  clear(): ArrayOperationClearData<T> {
    return {
      type: ArrayOperation.Clear,
      array: { next: [], prev: this.items },
    };
  }

  private getItemIndexById(id: ArrayOperationBuilderItemId): number {
    return this.items.findIndex((item) => item.id === id);
  }

  private getItemById(id: ArrayOperationBuilderItemId): T | undefined {
    return this.items.find((item) => item.id === id);
  }
}
