type Id = number | string;

interface Item {
  id: Id;
  [key: string]: any;
}

export type Value = Item[];

export enum ActionType {
  Clear = 'clear',
  Add = 'add',
  Delete = 'delete',
}

export interface ActionClear {
  type: ActionType.Clear;
}

export interface ActionMutation {
  type: ActionType.Add | ActionType.Delete;
  item: Item;
}

export default class ArrayValueManager {
  public static arrayToValue(
    value: Value,
    isSingleValue = false,
    isNullEmptyValue = false,
  ): Value | Item | null | undefined {
    if (!isSingleValue) {
      return value;
    }

    if (Array.isArray(value) && value[0] != null) {
      return value[0];
    }

    if (isNullEmptyValue) {
      return null;
    }

    return undefined;
  }

  public static valueToArray(value: Value | Item | null | undefined, isSingleValue = false): Value {
    if (Array.isArray(value)) {
      return value;
    }

    if (isSingleValue && value) {
      return [value];
    }

    return [];
  }

  public static createAction(oldValue: Value, newValue: Value): ActionMutation {
    const map = new Map<Id, ActionMutation>();

    if (Array.isArray(oldValue)) {
      oldValue.forEach(item => {
        map.set(item.id, { type: ActionType.Delete, item });
      });
    }

    newValue.forEach(item => {
      if (map.has(item.id)) {
        map.delete(item.id);
      } else {
        map.set(item.id, { type: ActionType.Add, item });
      }
    });

    return map.values().next().value;
  }

  private value: Value = [];

  public constructor(value?: Value) {
    this.set(value);
  }

  public getValue(): Value {
    return this.value;
  }

  public set(value?: Value): Value {
    if (Array.isArray(value)) {
      this.value = value;
    }

    return this.value;
  }

  public hasById(id: Id): boolean {
    return !!this.value.find(item => item.id === id);
  }

  public has(item: Item): boolean {
    return this.hasById(item.id);
  }

  public add(item: Item): Value {
    if (!this.has(item)) {
      this.value = this.value.concat(item);
    }
    return this.value;
  }

  public remove(item: Item): Value {
    return this.removeById(item.id);
  }

  public removeById(id: Id): Value {
    this.value = this.value.filter(i => i.id !== id);
    return this.value;
  }
}
