import { Subject, SubscriptionLike, Subscription, merge } from 'rxjs';
import { filter, tap, throttleTime } from 'rxjs/operators';

export interface BatchSubjectOptions {
  maxBufferSize?: number;
  throttleTime?: number;
}

export class BatchSubject<T> implements SubscriptionLike {
  static MAX_BUFFER_SIZE = 30;
  static THROTTLE_TIME = 5 * 1000;

  private innerSubject = new Subject<T>();
  private outerSubject = new Subject<T[]>();

  private subscription = new Subscription();

  private readonly options: Required<BatchSubjectOptions> = {
    maxBufferSize: BatchSubject.MAX_BUFFER_SIZE,
    throttleTime: BatchSubject.THROTTLE_TIME,
  };

  private buffer: T[] = [];

  public outerObservable = this.outerSubject.asObservable();

  constructor(options?: BatchSubjectOptions) {
    this.options = { ...this.options, ...options };

    this.subscription.add(this.innerSubject);
    this.subscription.add(this.outerSubject);

    this.initWatchers();
  }

  next(value?: T) {
    this.innerSubject.next(value);
  }

  get closed() {
    return this.innerSubject.closed;
  }

  unsubscribe(): void {
    this.subscription.unsubscribe();
  }

  private initWatchers() {
    const addItems = this.innerSubject.pipe(
      filter((value) => value != null),
      tap((value) => this.buffer.push(value)),
      tap(this.disposeBufferIfEnoughSize),
    );

    const throttle = this.innerSubject.pipe(
      throttleTime(this.options.throttleTime, undefined, { trailing: true }),
      tap(this.disposeBufferIfNotEmpty),
    );

    this.subscription.add(merge(addItems, throttle).subscribe());
  }

  private disposeBufferIfEnoughSize = () => {
    if (this.buffer.length !== this.options.maxBufferSize) {
      return;
    }

    this.disposeBufferIfNotEmpty();
  };

  private disposeBufferIfNotEmpty = () => {
    if (!this.buffer.length) {
      return;
    }

    this.outerSubject.next(this.buffer);
    this.buffer = [];
  };
}
