import { makeAutoObservable } from 'mobx';
import { LinkedList } from './LinkedList';
import { LinkedListNode, MutableLinkedListNode } from './LinkedListNode';
import { LinkedListNodeImpl } from './LinkedListNodeImpl';
import { xor } from '../xor';

export class LinkedListImpl<T> implements LinkedList<T> {
  private _count = 0;
  private _first: LinkedListNode<T> | undefined;
  private _last: LinkedListNode<T> | undefined;

  constructor(value?: Iterable<T> | ArrayLike<T>) {
    makeAutoObservable(this);

    if (!value) {
      return;
    }

    Array.from(value).forEach((item) => this.addLast(item));
  }

  get first(): LinkedListNode<T> | undefined {
    return this._first;
  }

  get last(): LinkedListNode<T> | undefined {
    return this._last;
  }

  get count(): number {
    return this._count;
  }

  addLast(newNodeOrValue: LinkedListNode<T> | T) {
    const node = this.convertToNode(newNodeOrValue) as MutableLinkedListNode<T>;

    node.list = this;
    node.previous = this._last;

    if (this._last) {
      (this._last as MutableLinkedListNode<T>).next = node;
    }

    this._last = node;
    this._count += 1;

    this.fixEdgeLinksAfterChange();
  }

  addFirst(newNodeOrValue: LinkedListNode<T> | T) {
    const node = this.convertToNode(newNodeOrValue) as MutableLinkedListNode<T>;

    node.list = this;
    node.next = this._first;

    if (this._first) {
      (this._first as MutableLinkedListNode<T>).previous = node;
    }

    this._first = node;
    this._count += 1;

    this.fixEdgeLinksAfterChange();
  }

  addAfter(currentNode: LinkedListNode<T>, newNodeOrValue: LinkedListNode<T> | T) {
    const node = this.convertToNode(newNodeOrValue) as MutableLinkedListNode<T>;
    const { next } = currentNode;

    if (next) {
      (next as MutableLinkedListNode<T>).previous = node;
    }

    node.list = this;
    node.previous = currentNode;
    node.next = next;

    this._count += 1;

    (currentNode as MutableLinkedListNode<T>).next = node;
  }

  clear() {
    while (this._first) {
      this.removeNode(this._first);
    }
  }

  removeNode(node: LinkedListNode<T>) {
    if (node.list !== this) {
      return;
    }

    const { previous, next } = node;

    if (previous) {
      (previous as MutableLinkedListNode<T>).next = next;
    }

    if (next) {
      (next as MutableLinkedListNode<T>).previous = previous;
    }

    const mutableNode = node as MutableLinkedListNode<T>;
    mutableNode.list = undefined;
    mutableNode.next = undefined;
    mutableNode.previous = undefined;

    this._count -= 1;
  }

  get asArray() {
    return this.toArray();
  }

  toArray() {
    const result: T[] = [];

    let currentNode = this._first;
    while (currentNode) {
      result.push(currentNode.value);
      currentNode = currentNode.next;
    }

    return result;
  }

  private fixEdgeLinksAfterChange() {
    if (xor(this._last, this._first)) {
      this._last = this._first = this._last || this._first;
    }
  }

  private convertToNode(newNodeOrValue: LinkedListNode<T> | T): LinkedListNode<T> {
    if (newNodeOrValue instanceof LinkedListNodeImpl) {
      return newNodeOrValue;
    }

    return new LinkedListNodeImpl<T>(newNodeOrValue as never);
  }
}
