import { TabMutexId } from './TabMutexId';
import { delay } from '../delay';

export interface TabMutexOptions {
  id: string | number;
  task: () => void;
  storage?: Storage;
}

export class TabMutex {
  public static readonly UNIQ_TAB_ID = Math.random().toString(36);

  public static readonly DELAY_FOR_STORAGE_UPDATE_MS = 100;

  public static createAndRun(options: TabMutexOptions) {
    const tabMutex = new TabMutex(options);
    tabMutex.run();

    return tabMutex;
  }

  private readonly mutexId: string;
  private readonly options: Required<TabMutexOptions>;

  constructor(options: TabMutexOptions) {
    this.options = { storage: localStorage, ...options };

    this.mutexId = new TabMutexId(this.options.id).toString();
  }

  public run() {
    this.tryRunTask();
  }

  private async tryRunTask() {
    if (this.isTaskRunningOnSomeTab()) {
      return;
    }

    this.tryHandleMutex();

    await delay(TabMutex.DELAY_FOR_STORAGE_UPDATE_MS);

    if (this.isSuccessCatchMutexForRunTask()) {
      this.options.task();
    }
  }

  private isTaskRunningOnSomeTab() {
    return this.options.storage[this.mutexId];
  }

  private tryHandleMutex() {
    this.options.storage[this.mutexId] = TabMutex.UNIQ_TAB_ID;
  }

  private isSuccessCatchMutexForRunTask() {
    return this.options.storage[this.mutexId] === TabMutex.UNIQ_TAB_ID;
  }
}
