import { TabMutexId } from './TabMutexId';

export class TabMutexCleaner {
  public static readonly LOCALSTORAGE_POLLING_DELAY_MS = 60 * 1000;
  public static readonly MUTEX_LIFETIME_MS = 60 * 1000;

  private mutexIdToTimeFoundInLocalStorage = new Map<string, number>();
  private timer?: number;

  constructor(private storage: Storage = window.localStorage) {
    this.checkLocalStorage = this.checkLocalStorage.bind(this);
  }

  run() {
    this.timer = window.setInterval(
      this.checkLocalStorage,
      TabMutexCleaner.LOCALSTORAGE_POLLING_DELAY_MS,
    );
  }

  private checkLocalStorage() {
    const keys = Object.keys(this.storage);

    this.cleanupForMutexRemovedByOutside(keys);
    this.updateMutex(keys);
  }

  destroy() {
    window.clearInterval(this.timer);
  }

  private cleanupForMutexRemovedByOutside(keys: string[]) {
    const set = new Set(keys);

    for (let [key] of this.mutexIdToTimeFoundInLocalStorage) {
      if (!set.has(key)) {
        this.mutexIdToTimeFoundInLocalStorage.delete(key);
      }
    }
  }

  private updateMutex(keys: string[]) {
    keys.forEach((key) => {
      if (this.canDeleteMutex(key)) {
        this.mutexIdToTimeFoundInLocalStorage.delete(key);
        this.storage.removeItem(key);
        return;
      }

      if (!key.startsWith(TabMutexId.PREFIX)) {
        return;
      }

      if (!this.mutexIdToTimeFoundInLocalStorage.has(key)) {
        this.mutexIdToTimeFoundInLocalStorage.set(key, Number(new Date()));
      }
    });
  }

  private canDeleteMutex(id: string) {
    return (
      this.mutexIdToTimeFoundInLocalStorage.has(id) &&
      Number(new Date()) - this.mutexIdToTimeFoundInLocalStorage.get(id)! >
        TabMutexCleaner.MUTEX_LIFETIME_MS
    );
  }
}
