/**
 * HOC - пробрасывает в props свойство destroyOnUnmount
 * небходимо для корретктной работы нескольких форм (redux-form) с одинаковым "form"
 * только последний компонент должен иметь флаг destroyOnUnmount === true
 * и после разрушения подчищать redux state
 **/
import React from 'react';
import PropTypes from 'prop-types';
import hoistNonReactStatic from 'hoist-non-react-statics';

const destroyDriver = config => (WrappedComponent) => {
  const componentName = WrappedComponent.displayName || WrappedComponent.name;

  class EnhancedComponent extends React.Component {
    static forceUpdate(key) {
      EnhancedComponent.mountInstance[key].forEach((instance) => { instance.forceUpdate(); });
    }

    componentDidMount() {
      this.addInstance(this.getSlug());
    }

    componentWillReceiveProps(nextProps) {
      const nextSlug = nextProps[this.props.slugName];
      const currentSlug = this.getSlug();

      if (nextSlug !== currentSlug) {
        this.addInstance(nextSlug);
        this.removeInstance(currentSlug);
      }
    }

    componentWillUnmount() {
      this.removeInstance(this.getSlug());
    }

    getSlug() {
      return this.props[this.props.slugName];
    }

    addInstance(slug) {
      if (!EnhancedComponent.mountInstance[slug]) {
        EnhancedComponent.mountInstance[slug] = [];
      }

      const list = EnhancedComponent.mountInstance[slug];
      list.push(this);

      EnhancedComponent.forceUpdate(slug);
    }

    removeInstance(slug) {
      let list = EnhancedComponent.mountInstance[slug];
      list = list.filter(
        instance => (instance !== this),
      );

      if (!list.length) {
        delete EnhancedComponent.mountInstance[slug];
      } else {
        EnhancedComponent.mountInstance[slug] = list;
        EnhancedComponent.forceUpdate(slug);
      }
    }

    render() {
      const list = EnhancedComponent.mountInstance[this.getSlug()];
      // разрушаем redux state компонента если он остался один на странице
      const destroyOnUnmount = !!(list && (list.length === 1));
      return React.createElement(WrappedComponent, {
        ...this.props,
        destroyOnUnmount,
        ref: (c) => { this.__wrappedComponent = ((c && c.__wrappedComponent) ? c.__wrappedComponent : c); }, // eslint-disable-line no-underscore-dangle
      });
    }
  }

  EnhancedComponent.defaultProps = {
    slugName: 'form',
    ...config,
  };
  EnhancedComponent.propTypes = {
    slugName: PropTypes.string,
  };
  EnhancedComponent.mountInstance = {};
  EnhancedComponent.displayName = `destroyDriver(${componentName})`;

  return hoistNonReactStatic(EnhancedComponent, WrappedComponent);
};

export default destroyDriver;
