import { useState, useEffect } from 'react';
import { logger } from 'services/Logger';
import { get } from 'api/common';
import Promise from 'bluebird';
import { bunkersNodeKeys } from './bunkersNodeKeys';
import { mergeConfigs } from './useBunker.utils/mergeConfigs';
import { BunkerState, FlagDependantConfig } from './useBunker.types';

export const useBunker = <T extends FlagDependantConfig>(nodePaths: string[]) => {
  const [bunkerData, setBunkerData] = useState<BunkerState<T>>({
    isFetched: false,
    data: undefined,
  });

  useEffect(() => {
    const promises = nodePaths
      .filter((path) => bunkersNodeKeys.has(path))
      .map((path) => {
        return get({ url: `v0/bunker/cat?node=${path}`, host: '/' });
      });

    if (promises.length > 0) {
      const request = Promise.all<T>(promises)
        .then((data) => {
          setBunkerData({
            isFetched: true,
            data: mergeConfigs(data),
          });
        })
        .catch((error) => {
          setBunkerData({ isFetched: true, data: undefined });
          logger.reportError(error);
        });
      return () => request.cancel();
    }

    if (!bunkerData.isFetched) {
      setBunkerData({ isFetched: true, data: undefined });
    }
  }, [bunkerData.isFetched, nodePaths]);

  return bunkerData;
};
