import { useEffect, useState, useRef } from 'react';

const allowedTypes = ['Files', 'application/json'];

const eventHasAllowedTypes = (event: DragEvent) => {
  const types = event.dataTransfer?.types || [];

  return types.some((type) => allowedTypes.includes(type));
};

export const useFileDragState = () => {
  const [active, setActive] = useState(false);

  const counter = useRef(0);

  const handleDrag = (event: DragEvent) => {
    if (!eventHasAllowedTypes(event)) return;

    // При перетаскивании, события dragenter и dragleave срабатывают на элементах внутри document
    counter.current += event.type === 'dragenter' ? 1 : -1;

    setActive(counter.current > 0);
  };

  const handleDrop = (event: DragEvent) => {
    event.preventDefault();
    setActive(false);
    counter.current = 0;
  };

  useEffect(() => {
    document.addEventListener('dragenter', handleDrag);
    document.addEventListener('dragleave', handleDrag);
    document.addEventListener('drop', handleDrop);

    return () => {
      document.removeEventListener('dragenter', handleDrag);
      document.removeEventListener('dragleave', handleDrag);
      document.removeEventListener('drop', handleDrop);
    };
  }, []);

  return active;
};
