import { useEffect } from 'react';
import { KeyCallbacks } from './useKeyboard.types';

export const useKeyboard = (
  keys: KeyCallbacks,
  isListening: boolean = true,
  deps: unknown[] = [],
) => {
  const {
    onUp,
    onLeft,
    onDown,
    onRight,
    onEnter,
    onSpace,
    onEsc,
    onTab,
    onUpCapture,
    onLeftCapture,
    onDownCapture,
    onRightCapture,
    onEnterCapture,
    onSpaceCapture,
    onEscCapture,
    onTabCapture,
    onBackspace,
    onBackspaceCapture,
  } = keys;

  useEffect(() => {
    if (!isListening) {
      return;
    }

    const bubbleMap = {
      ArrowUp: onUp,
      ArrowLeft: onLeft,
      ArrowDown: onDown,
      ArrowRight: onRight,
      Enter: onEnter,
      ' ': onSpace,
      Escape: onEsc,
      Tab: onTab,
      Backspace: onBackspace,
    };

    const captureMap = {
      ArrowUp: onUpCapture,
      ArrowLeft: onLeftCapture,
      ArrowDown: onDownCapture,
      ArrowRight: onRightCapture,
      Enter: onEnterCapture,
      ' ': onSpaceCapture,
      Escape: onEscCapture,
      Tab: onTabCapture,
      Backspace: onBackspaceCapture,
    };

    const handleKeyDown = (event: KeyboardEvent) => {
      if (bubbleMap[event.key]) {
        bubbleMap[event.key](event);
      }
    };

    const handleKeyDownCapture = (event: KeyboardEvent) => {
      if (captureMap[event.key]) {
        captureMap[event.key](event);
      }
    };

    document.addEventListener('keydown', handleKeyDown);
    document.addEventListener('keydown', handleKeyDownCapture, true);
    return () => {
      document.removeEventListener('keydown', handleKeyDown);
      document.removeEventListener('keydown', handleKeyDownCapture, true);
    };
  }, [...deps, isListening]);
};
