import { useRef, RefObject, useCallback, MouseEvent } from 'react';

export const useNullTextSelectionClick = (
  ref: RefObject<HTMLElement | undefined | null>,
  onClick?: (event: MouseEvent) => void,
) => {
  const isMouseInside = useRef(false);
  const isMouseDown = useRef(false);
  const willHandlerCalled = useRef(false);

  const handleMouseEnter = useCallback(() => {
    isMouseInside.current = true;
  }, []);

  const handleMouseLeave = useCallback(() => {
    isMouseInside.current = false;
  }, []);

  const handleMouseDown = useCallback(() => {
    window.getSelection()?.removeAllRanges();
    if (isMouseInside.current) {
      isMouseDown.current = true;
    }
  }, []);

  const handleMouseUp = useCallback(() => {
    if (!isMouseDown.current) {
      return;
    }

    isMouseDown.current = false;
    const selection = window.getSelection();
    if (selection && !selection.isCollapsed && ref.current?.contains(selection.anchorNode)) {
      return;
    }
    willHandlerCalled.current = true;
    setTimeout(() => {
      willHandlerCalled.current = false;
      // хак для ситуации, когда на click event делают stopPropagation.
      // без хака обработчик будет вызываться на следующем таргете, даже если он
      // не пройдет по условиям.
    }, 0);
  }, []);

  const handleClick = useCallback(
    (event: MouseEvent) => {
      if (!willHandlerCalled.current) {
        willHandlerCalled.current = false;
        return;
      }

      onClick?.(event);
      willHandlerCalled.current = false;
    },
    [onClick],
  );

  return {
    onMouseEnter: handleMouseEnter,
    onMouseLeave: handleMouseLeave,
    onMouseDown: handleMouseDown,
    onMouseUp: handleMouseUp,
    onClick: handleClick,
  };
};
