import { ReducersMapObject, Reducer, Action, AnyAction } from '@reduxjs/toolkit';

function getUndefinedStateErrorMessage(key: string, action: Action) {
  const actionType = action && action.type;
  const actionDescription = (actionType && `action "${String(actionType)}"`) || 'an action';

  return (
    `Given ${actionDescription}, reducer "${key}" returned undefined. ` +
    `To ignore an action, you must explicitly return the previous state. ` +
    `If you want this reducer to hold no value, you can return null instead of undefined.`
  );
}

// eslint-disable-next-line @typescript-eslint/no-explicit-any
function combineReducers<S>(reducers: ReducersMapObject<S, any>): Reducer<S>;
function combineReducers<S, A extends Action = AnyAction>(
  reducers: ReducersMapObject<S, A>,
): Reducer<S, A>;

function combineReducers(reducers) {
  const reducerKeys = Object.keys(reducers);

  const filteredReducers = reducerKeys.reduce((result, key) => {
    if (typeof reducers[key] === 'function') {
      result[key] = reducers[key];
    }

    return result;
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
  }, {} as any);

  const filteredReducerKeys = Object.keys(filteredReducers);

  return (state = {}, action) => {
    let hasChanged = false;

    const nextState = filteredReducerKeys.reduce((result, key) => {
      const nextStateForKey = reducers[key](state[key], action);

      if (typeof nextStateForKey === 'undefined') {
        const errorMessage = getUndefinedStateErrorMessage(key, action);
        throw new Error(errorMessage);
      }

      result[key] = nextStateForKey;
      hasChanged = hasChanged || result[key] !== state[key];
      return result;
    }, {});

    if (filteredReducerKeys.length !== Object.keys(state).length) {
      hasChanged = true;
    }

    return hasChanged ? nextState : state;
  };
}

export default combineReducers;
