/* eslint-disable import/no-extraneous-dependencies */
import path from 'path';
import webpack from 'webpack';
import CircularDependencyPlugin from 'circular-dependency-plugin';
import { RetryChunkLoadPlugin } from 'webpack-retry-chunk-load-plugin';
import { threadLoader } from './thread-loader';

interface Props {
  env?: 'development' | 'production';
  /** устанавливает __DEV_FEATURE__ для прода на клиенте */
  devFeature?: boolean;
  /** корневая папка проекта */
  rootDirName?: string;
  /** корневая папка файлов сборки */
  distPath?: string;
  // eslint-disable-next-line @typescript-eslint/no-explicit-any
  features?: any;
}

const isStorybook = process.env.STORYBOOK_ENV === 'test';
const ThreadLoader = isStorybook ? [] : [threadLoader];

// https://webpack.js.org/configuration/module/#resolvefullyspecified
const webpack5esmInteropRule = {
  test: /\.m?js/,
  resolve: {
    fullySpecified: false,
  },
};

export default function config(options: Props = {}): webpack.Configuration {
  return {
    output: {
      path: options.distPath,
    },
    externals: [
      {
        '@yandex-data-ui/navigation': {},
      },
    ],
    module: {
      rules: [
        webpack5esmInteropRule,
        {
          test: /\.worker\.[jt]s$/,
          use: [
            {
              loader: 'worker-loader',
              options: {
                inline: 'no-fallback',
              },
            },
            {
              loader: 'babel-loader',
            },
          ],
        },
        {
          test: /\.sharedworker\.[jt]s$/,
          use: [
            {
              loader: './webpack/loaders/sharedworker-loader',
              options: {
                worker: {
                  type: 'SharedWorker',
                },
              },
            },
          ],
        },
        {
          test: /\.(js|ts)x?$/,
          use: [
            ...ThreadLoader,
            {
              loader: 'babel-loader',
            },
          ],
          exclude: /node_modules/,
          include: options.rootDirName,
        },
        {
          test: /\.(png|gif)$/,
          type: 'asset/inline',
          parser: {
            dataUrlCondition: {
              maxSize: 10000,
            },
          },
        },
        {
          test: /\.inline\.svg/,
          use: 'svg-url-loader',
        },
        {
          test: /\.react\.svg$/,
          use: [
            {
              loader: 'babel-loader',
            },
            {
              loader: 'svg-sprite-loader',
              options: {
                symbolId: '[name].[hash:base64:5]',
                runtimeGenerator: require.resolve('./svg-to-icon-component-runtime-generator'),
                runtimeOptions: {
                  iconModule: './webpack/icon.tsx',
                },
              },
            },
            {
              loader: 'svgo-loader',
            },
          ],
        },
        {
          test: /\.(svg|ttf|eot|woff|woff2|wav|mp3)/,
          exclude: /\.(inline|react)\.svg$/,
          type: 'asset/resource',
          generator: {
            filename: '[path][name][ext]',
          },
        },
      ],
    },
    resolve: {
      alias: {},
      modules: [
        path.resolve(options.rootDirName ?? '', 'src'),
        path.resolve(options.rootDirName ?? '', 'src/entry'),
        'node_modules',
        // path.resolve(options.rootDirName, 'src/styles'),
      ],
      extensions: ['.ts', '.tsx', '.js', '.jsx', '.json', '.css', '.scss'],
      enforceExtension: false,
    },
    plugins: [
      new webpack.DefinePlugin({
        'process.env': {
          NODE_ENV: JSON.stringify(options.env),
        },
        __DEV_FEATURE__: JSON.stringify(options.devFeature || options.env === 'development'),
        __FEATURES__: (options.features || '')
          .split(',')
          .map((item) => item.trim())
          .reduce((ac, key) => ({ ...ac, [key]: true }), {}),
      }),
      new webpack.ContextReplacementPlugin(/node_modules\/moment\/locale/, /ru|en-gb/),
      new webpack.LoaderOptionsPlugin({
        debug: true,
        options: {
          context: options.rootDirName,
        },
      }),
      new RetryChunkLoadPlugin({
        // optional value to set the amount of time in milliseconds before trying to load the chunk again. Default is 0
        retryDelay: 3000,
        // optional value to set the maximum number of retries to load the chunk. Default is 1
        maxRetries: 3,
        // optional code to be executed in the browser context if after all retries chunk is not loaded.
        // if not set - nothing will happen and error will be returned to the chunk loader.
        lastResortScript: `(${function(chunkId: string) {
          const error = new Error();
          error.name = 'ChunkLoadError';
          // @ts-ignore
          error.userMessage =
            'Не получилось загрузить модуль приложения. Попробуйте обновить страницу. Если проблема остается, напишите в поддержку.';
          // @ts-ignore
          error.chunkId = chunkId;

          throw error;
        }.toString()})(chunkId);`,
      }),
      false && // TODO: https://st.yandex-team.ru/CRM-11158
        new CircularDependencyPlugin({
          // exclude detection of files based on a RegExp
          exclude: /node_modules/,
          // add errors to webpack instead of warnings
          failOnError: true,
          // allow import cycles that include an asyncronous import,
          // e.g. via import(/* webpackMode: "weak" */ './file.js')
          allowAsyncCycles: false,
          // set the current working directory for displaying module paths
          cwd: process.cwd(),
        }),
    ].filter(Boolean),
  };
}
