/* eslint-disable import/no-extraneous-dependencies */
import path from 'path';
import HtmlWebpackPlugin from 'html-webpack-plugin';
import { BundleAnalyzerPlugin } from 'webpack-bundle-analyzer';
import webpack from 'webpack';
import { merge } from 'webpack-merge';
import RumInlinePlugin from './plugins/rum-inline-plugin';
import commonConfig from './common.config';

const BACKEND_INSTANCES = {
  dev: 'https://develop.crm-test.yandex-team.ru',
  test: 'https://crm-test.yandex-team.ru',
  mars: 'https://mars.crm.yandex-team.ru',
  prod: 'https://crm.yandex-team.ru',
};

const BundleAnalyzer = process.env.ANALYZE
  ? [
      new BundleAnalyzerPlugin({
        analyzerMode: 'static',
      }),
    ]
  : [];

interface DevConfigOptions {
  server: string;
  serverUrl: string;
}

const devConfig = (
  options: DevConfigOptions = { server: 'test', serverUrl: '' },
): webpack.Configuration => ({
  mode: 'development',
  devtool: 'cheap-module-source-map',
  entry: {
    rumInline: ['./src/services/Rum/config/inline.js'],
    rumBundle: ['./src/services/Rum/config/bundle.js'],
    entry: ['./src/entry/'],
  },
  module: {
    rules: [
      {
        test: /\.(scss|modules\.scss|module\.css)$/,
        use: [
          {
            loader: 'style-loader',
          },
          {
            loader: 'css-loader',
            options: {
              modules: {
                localIdentName: '[path]___[name]__[local]___[hash:base64:5]',
              },
              importLoaders: 1,
            },
          },
          {
            loader: 'postcss-loader',
          },
        ],
      },
      {
        test: /\.(css|raw\.scss)$/,
        exclude: /\.module\.css$/,
        use: [
          {
            loader: 'style-loader',
          },
          {
            loader: 'css-loader',
            options: {
              importLoaders: 1,
            },
          },
          {
            loader: 'postcss-loader',
          },
        ],
      },
    ],
  },
  output: {
    filename: 'dist/bundle.[name].js',
    chunkFilename: '[id].js',
  },
  cache: {
    type: 'filesystem',
    buildDependencies: {
      config: [__filename],
    },
  },
  optimization: {
    moduleIds: 'named',
    sideEffects: false,
  },
  stats: {
    modules: false,
  },
  plugins: [
    new RumInlinePlugin(),
    new HtmlWebpackPlugin({
      template: 'src/assets/index.csharp.ejs',
      excludeChunks: ['rumInline'],
    }),
    ...BundleAnalyzer,
  ],
  devServer: {
    server: {
      type: 'https',
    },
    client: {
      overlay: false,
    },
    static: [
      {
        directory: path.resolve(__dirname, '../vendor'),
        publicPath: '/vendor',
      },
      {
        directory: path.resolve(__dirname, '../src/components/RichHtmlEditor/ckeditor'),
        publicPath: '/vendor/ckeditor',
      },
      {
        directory: path.resolve(__dirname, '../img'),
        publicPath: '/img',
      },
    ],
    host: 'localhost.yandex-team.ru',
    port: 443,
    proxy: [
      {
        context: ['**', '!/vendor/**', '!/img/**', '!/src/components/RichHtmlEditor/ckeditor/**'],
        headers: {
          'Access-Control-Allow-Origin': '*',
        },
        target: options.serverUrl || BACKEND_INSTANCES[options.server] || BACKEND_INSTANCES.test,
        secure: false,
        changeOrigin: true,
      },
    ],
  },
});

export default (...args) => merge(commonConfig(...args), devConfig(...args));
