/* eslint-disable import/no-extraneous-dependencies */
import webpack from 'webpack';
import { merge } from 'webpack-merge';
import AssetsPlugin from 'assets-webpack-plugin';
import HtmlWebpackPlugin from 'html-webpack-plugin';
import CopyPlugin from 'copy-webpack-plugin';
import rimraf from 'rimraf';
import MiniCssExtractPlugin from 'mini-css-extract-plugin';
import CssoWebpackPlugin from 'csso-webpack-plugin';
import commonConfig from './common.config';
import RumInlinePlugin from './plugins/rum-inline-plugin';

interface Props {
  env?: 'development' | 'production';
  version?: string;
  /** корневая папка файлов сборки */
  distPath?: string;
  /** устанавливает __DEV_FEATURE__ для прода на клиенте */
  devFeature?: boolean;
  /** корневая папка проекта */
  rootDirName?: string;
  publicPath?: string;
  sourceMapPublicPath?: string;
  newDeploy?: string;
}

const prodConfig = (options: Props = {}): webpack.Configuration => {
  const { publicPath, newDeploy, sourceMapPublicPath } = options;

  return {
    mode: 'production',
    devtool: newDeploy ? false : 'source-map',
    entry: {
      rumInline: './src/services/Rum/config/inline.js',
      rumBundle: './src/services/Rum/config/bundle.js',
      entry: './src/entry/',
    },
    module: {
      rules: [
        {
          test: /\.(scss|modules\.scss|module\.css)$/,
          use: [
            {
              loader: MiniCssExtractPlugin.loader,
            },
            {
              loader: 'css-loader',
              options: {
                modules: {
                  localIdentName: '[hash:base64]',
                },
                importLoaders: 1,
              },
            },
            {
              loader: 'postcss-loader',
            },
          ],
        },
        {
          test: /\.(css|raw\.scss)$/,
          exclude: /\.module\.css$/,
          use: [
            {
              loader: MiniCssExtractPlugin.loader,
            },
            {
              loader: 'css-loader',
              options: {
                importLoaders: 1,
              },
            },
            {
              loader: 'postcss-loader',
            },
          ],
        },
      ],
    },
    output: {
      filename: 'bundle.[name].[chunkhash].js',
      chunkFilename: '[name].[chunkhash].js',
      publicPath,
    },
    cache: {
      type: 'filesystem',
      buildDependencies: {
        config: [__filename],
      },
    },
    optimization: {
      sideEffects: false,
      splitChunks: {
        chunks: 'all',
        cacheGroups: {
          defaultVendors: {
            test: /[\\/]node_modules[\\/]/,
            name: 'vendors',
            priority: -10,
          },
          asyncVendors: {
            chunks: 'async',
            test: /[\\/]node_modules[\\/]/,
            name: 'asyncVendors',
            priority: -5,
          },
          default: {
            minChunks: 2,
            priority: -20,
            reuseExistingChunk: true,
          },
        },
      },
      runtimeChunk: {
        name: 'manifest',
      },
    },
    plugins: [
      {
        apply: (compiler) => {
          rimraf.sync(compiler.options.output.path);
        },
      },
      sourceMapPublicPath &&
        new webpack.SourceMapDevToolPlugin({
          append: `\n//# sourceMappingURL=${sourceMapPublicPath}[url]`,
          filename: '[file].map',
        }),
      new MiniCssExtractPlugin({
        filename: '[name].[chunkhash].css',
        chunkFilename: '[id].[chunkhash].css',
        experimentalUseImportModule: true,
      }),
      new CssoWebpackPlugin(),
      new AssetsPlugin({
        filename: 'assets.json',
        path: options.distPath,
      }),
      new HtmlWebpackPlugin({
        filename: '../../index.html',
        template: publicPath ? 'src/assets/index.node.ejs' : 'src/assets/index.csharp.ejs',
        appVersion: options.version,
        publicPath: publicPath ? publicPath : 'auto',
        chunks: ['rumBundle', 'vendors', 'manifest', 'main', 'entry'],
      }),
      new CopyPlugin({
        patterns: [
          { from: 'src/components/RichHtmlEditor/ckeditor/skins', to: 'vendor/ckeditor/skins' },
        ],
      }),
      new RumInlinePlugin(options.version),
    ].filter(Boolean),
  };
};

export default (...args) => merge(commonConfig(...args), prodConfig(...args));
