import os
import sys
import threading
import logging
import time

from flask import Flask, jsonify

from library.python.deploy_formatter import DeployFormatter

from .logbroker import ReadMessageProcessor, start_read_session
from .ora_client import Oracle
from .stat import Stat


log_formatter = DeployFormatter()

errors_handler = logging.StreamHandler(sys.stderr)
errors_handler.setLevel(logging.WARN)
errors_handler.setFormatter(log_formatter)
others_handler = logging.StreamHandler(sys.stdout)
others_handler.setLevel(logging.DEBUG)
others_handler.addFilter(lambda record: record.levelno <= logging.INFO)
others_handler.setFormatter(log_formatter)

root = logging.getLogger()
root.setLevel(logging.INFO)
root.addHandler(errors_handler)
root.addHandler(others_handler)

logger = logging.getLogger(__name__)
logger.setLevel(logging.DEBUG)

app = Flask(__name__)


class OracleProcessing(ReadMessageProcessor):
    def __init__(self, logbroker_host):
        self.ora = Oracle(Config.connectionString)
        self.logbroker_ep = self.extract_ep(logbroker_host)

    def process(self, messages):
        count = len(messages)
        if not count:
            logger.debug('No messages to insert')
            return

        params = list(
            map(lambda m: (self.logbroker_ep, m.meta.source_id, m.meta.seq_no, m.data.decode('UTF-8')), messages))
        self.ora.nonQueryBatch(
            ("merge into bpmonline.ext_lb_{} s using (select :1 endpoint, :2 src, :3 seq_id from dual) t "
             "on (s.endpoint = t.endpoint and s.src = t.src and s.seq_id = t.seq_id) when not matched then "
             "insert (endpoint, src, seq_id, data, import_dt) values (t.endpoint, t.src, t.seq_id, :4, sysdate)").format(
                Config.lbTable), params)

        logger.debug(f'Number of messages inserted/merged: {count}')

    def extract_ep(self, logbroker_host):
        return logbroker_host.split('.')[0]


class MessageProcessing(ReadMessageProcessor):

    def __init__(self, logbroker_host):
        self.logbroker_host = logbroker_host

    def process(self, messages):
        for message in messages:
            logger.debug("[%s] Process message: %s", self.logbroker_host, message)


class CommonProcessing(ReadMessageProcessor):
    def __init__(self, logbroker_host):
        self.oraProcessing = OracleProcessing(logbroker_host)
        self.msgProcessing = MessageProcessing(logbroker_host)

    def process(self, messages):
        self.msgProcessing.process(messages)
        self.oraProcessing.process(messages)


def main():
    hosts_str = os.environ.get("LB_HOSTS")
    hosts = hosts_str.split(';')
    logger.debug("hosts are: %s", hosts)
    threads = []
    flask_thread = threading.Thread(target=app.run, kwargs={
        "host": "::",
        "port": 80,
        "debug": False,
        "use_reloader": False
    })
    flask_thread.start()
    threads.append(flask_thread)

    for host in hosts:
        thread = threading.Thread(target=start_for_host, args=(host,), name=host)
        threads.append(thread)
        thread.start()
    input('Press ENTER to continue')
    logger.debug("exiting the main thread")


def start_for_host(logbroker_host):
    stat_prefix = os.environ.get("LB_UNISTAT_SIGNAL_PREFIX")
    prj_name = get_prj_name()
    stat = Stat(stat_prefix + logbroker_host.split('.')[0], prj_name)
    while True:
        try:
            logger.debug("Starting thread %s", logbroker_host)
            read_processor = CommonProcessing(logbroker_host)
            with start_read_session(logbroker_host, read_processor) as lb:
                logger.debug("Started thread %s", logbroker_host)
                while True:
                    read_messages = lb.read_message()
                    count = len(read_messages)
                    logger.debug(f"messages successfully read from {logbroker_host}: {count}")
                    stat.send_signal(1.0)
        except Exception as e:
            logger.exception("Catched exception {}".format(e))
            stat.send_signal(0.0)
        except:
            logger.error("Panic: unexpected error")
            stat.send_signal(0.0)
        finally:
            logger.debug("Waiting for restart")
            time.sleep(5)


def get_prj_name():
    if os.environ.get("DEPLOY_PROJECT_ID") is not None:
        return get_ydeploy_prj_name()
    elif os.environ.get("QLOUD_PROJECT") is not None:
        return get_qloud_prj_name()
    else:
        raise RuntimeError("Unknown deployment system")


def get_ydeploy_prj_name():
    return os.environ.get("DEPLOY_STAGE_ID") + "." + \
    os.environ.get("DEPLOY_UNIT_ID") + "." + \
    os.environ.get("DEPLOY_BOX_ID")


def get_qloud_prj_name():
    return os.environ.get("QLOUD_PROJECT") + "." + \
    os.environ.get("QLOUD_APPLICATION") + "." + \
    os.environ.get("QLOUD_ENVIRONMENT")


class Config:
    connectionString = str(os.environ.get("CRM_ORA_USER")) + "@" + str(os.environ.get("CRM_ORA_HOST"))
    lbTable = str(os.environ.get("CRM_LB_TABLE"))


@app.route('/stat')
def stat():
    stats = Stat.get_stats()
    logger.debug("Request for stats data: %s", stats)
    return jsonify(stats)
