import aiohttp
import datetime
from crm.supskills.common.deploy_logger import create_logger


logger = create_logger('bunker_logger')


class Bunker:
    def __init__(self, environment, time_delta):
        """
        self.cache = {'some/path': {'data': 'result of _collect_nodes', 'last_check': datetime.now()}, ...}
        """
        self.environment = environment
        self.time_delta = time_delta
        self.cache = {}
        self.bunker_host = 'http://bunker-api-dot.yandex.net/v1/cat?node=/direct-sup-skill/{}&version=latest' if \
            self.environment == 'testing' else 'http://bunker-api.yandex.net/v1/cat?node=/direct-sup-skill/{}'

    async def _update_cache(self, path):
        try:
            async with aiohttp.ClientSession() as session:
                async with session.get(self.bunker_host.format(path)) as response:
                    res = await response.json()
                    if res:
                        self.cache[path] = dict(last_check=datetime.datetime.now(), data=res)
        except Exception as e:
            logger.error('Error in bunker api response: {}, path {}'.format(e, path))

    def _is_update_required(self, path, node):
        value_path = self.cache.get(path)

        if not value_path:
            return True

        value_node = value_path['data'].get(node)

        if not value_node and node:
            return True

        path_last_check = value_path.get('last_check')
        return datetime.datetime.now() - path_last_check > self.time_delta

    async def get_node(self, path, node, skip_if_no_var=False):

        if self._is_update_required(path, node):
            await self._update_cache(path)

        try:
            return self.cache[path]['data'][node]

        except KeyError:
            if skip_if_no_var:
                return None
            else:
                logger.error('{}/{} is missing'.format(path, node))
                return 'Что-то сломалось.'

    async def get_all_nodes_in_path(self, path):

        if self._is_update_required(path, None):
            await self._update_cache(path)

        try:
            return self.cache[path]['data']

        except KeyError:
            return {}
