import aiohttp
from crm.supskills.common.direct_client.api_v4 import EnvType, DirectKeyError, DirectAPIError, DirectNetError
from crm.supskills.common.direct_client.services import Service, Ads, AdGroups, Campaigns, Changes, Feeds, TurboPages, \
    Keywords, BidModifiers, AudienceTargets, Clients
from crm.supskills.common.direct_client.structs.general import make_request_dict, from_dict
from crm.supskills.common.direct_client.structs.structs.ad_groups import AdGroupGetItem
from crm.supskills.common.direct_client.structs.structs.ads import AdGetItem
from crm.supskills.common.direct_client.structs.structs.audiencetargets import AudienceTargetGetItem
from crm.supskills.common.direct_client.structs.structs.bidmodifiers import BidModifierGetItem
from crm.supskills.common.direct_client.structs.structs.campaigns import CampaignGetItem
from crm.supskills.common.direct_client.structs.structs.changes import CheckResponseModified
from crm.supskills.common.direct_client.structs.structs.clients import ClientGetItem
from crm.supskills.common.direct_client.structs.structs.feeds import FeedGetItem
from crm.supskills.common.direct_client.structs.structs.keywords import KeywordGetItem
from crm.supskills.common.direct_client.structs.structs.turbopages import TurboPageGetItem
from crm.supskills.common.deploy_logger import create_logger

logger = create_logger('direct_api_v5_logger')


class Direct5:

    def __init__(self, env_type: EnvType = EnvType.testing, oauth_token: str = None):
        self.link_prefix = 'https://api.test.direct.yandex.ru:14443/json/v5/' \
            if env_type == EnvType.testing else 'https://ipv6.api.direct.yandex.ru/json/v5/'
        self.oauth_token = oauth_token

    def __make_link(self, from_which):
        return self.link_prefix + from_which

    def __make_headers(self, client_login: str):
        return {
            'Authorization': 'Bearer ' + self.oauth_token,
            'Client-Login': client_login,
            'Token-Type': 'persistent',
            'Accept-Language': 'ru'
        }

    async def __send_request(self, service: Service, parameters_data: dict, client_login: str) -> dict:
        try:
            async with aiohttp.ClientSession(headers=self.__make_headers(client_login), trust_env=True,
                                             connector=aiohttp.TCPConnector(verify_ssl=False)) as session:
                async with session.post(self.__make_link(service.link), json=parameters_data) as response:
                    return await response.json()
        except Exception as e:
            logger.debug('requests.post failed: %s', str(e))
            raise DirectNetError from e

    async def _make_request(self, params: dict, service: Service, client_login: str):  # -> service.result
        parameters_data: dict = make_request_dict(service, params)
        logger.info('Request to %s.', self.link_prefix + service.link)
        response: dict = await self.__send_request(service, parameters_data, client_login)
        logger.info('Response from %s, body is None: %s.', service.link, str(response is None))
        if isinstance(response, dict) and 'result' in response.keys():
            result_class: service.result = from_dict(service.result, response['result'])
            return result_class
        logger.debug('DirectAPIError: %s', str(response))
        raise DirectAPIError(f'Error: {str(response)}')

    async def get_single_campaign(self, client_login: str) -> int:
        result_class: Campaigns.result = await self._make_request({}, Campaigns, client_login)
        if result_class.Campaigns is not None and len(result_class.Campaigns) == 1:
            return result_class.Campaigns[0]
        logger.debug('DirectKeyError: result_class.Campaigns is None: %s.', str(result_class.Campaigns is None))
        raise DirectKeyError('Error: There is not a single campaign!')

    async def get_campaigns(self, client_login: str) -> [CampaignGetItem]:
        result_class: Campaigns.result = await self._make_request({}, Campaigns, client_login)
        return result_class.Campaigns

    async def get_campaigns_by_ids(self, client_login: str, campaign_ids: list[int]) -> [CampaignGetItem]:
        if not campaign_ids:
            return []
        result_class: Campaigns.result = await self._make_request(
            {'SelectionCriteria': {'Ids': campaign_ids}}, Campaigns, client_login)
        return result_class.Campaigns or []

    async def get_campaign(self, client_login: str, campaign_id: int) -> CampaignGetItem:
        result_class: Campaigns.result = await self._make_request(
            {'SelectionCriteria': {'Ids': [campaign_id]}}, Campaigns, client_login)
        if result_class.Campaigns is not None and len(result_class.Campaigns) == 1:
            return result_class.Campaigns[0]
        logger.debug('DirectKeyError: result_class.Campaigns is None: %s.', str(result_class.Campaigns is None))
        raise DirectKeyError('Error: Wrong Campaign Id!')

    async def get_ads(self, client_login: str, campaign_id: int) -> [AdGetItem]:
        result_class: Ads.result = await self._make_request(
            {'SelectionCriteria': {'CampaignIds': [campaign_id]}}, Ads, client_login)
        return result_class.Ads

    async def get_ads_by_ids(self, client_login: str, ad_ids: list[int]) -> [AdGetItem]:
        if not ad_ids:
            return []
        result_class: Ads.result = await self._make_request(
            {'SelectionCriteria': {'Ids': ad_ids}}, Ads, client_login)
        return result_class.Ads or []

    async def get_ad(self, client_login: str, ad_id: int) -> AdGetItem:
        result_class: Ads.result = await self._make_request(
            {'SelectionCriteria': {'Ids': [ad_id]}}, Ads, client_login)
        if result_class.Ads is not None and len(result_class.Ads) == 1:
            return result_class.Ads[0]
        logger.debug('DirectKeyError: result_class.Ads is None: %s.', str(result_class.Ads is None))
        raise DirectKeyError('Error: Wrong Ad Id!')

    async def get_ad_groups(self, client_login: str, campaign_id: int) -> [AdGroupGetItem]:
        result_class: AdGroups.result = await self._make_request(
            {'SelectionCriteria': {'CampaignIds': [campaign_id]}}, AdGroups, client_login)
        return result_class.AdGroups

    async def get_ad_groups_by_ids(self, client_login: str, ad_group_ids: list[int]) -> [AdGroupGetItem]:
        if not ad_group_ids:
            return []
        result_class: AdGroups.result = await self._make_request(
            {'SelectionCriteria': {'Ids': ad_group_ids}}, AdGroups, client_login)
        return result_class.AdGroups or []

    async def get_changes_in_campaign(self, client_login: str, campaign_id: int, timestamp: str) \
            -> CheckResponseModified:
        result_class: Changes.result = await self._make_request(
            {'CampaignIds': [campaign_id], 'Timestamp': timestamp}, Changes, client_login)
        return result_class.Modified

    async def get_feeds(self, client_login: str, feeds_ids: [int]) -> [FeedGetItem]:
        result_class: Feeds.result = await self._make_request(
            {'SelectionCriteria': {'Ids': feeds_ids}}, Feeds, client_login)
        return result_class.Feeds

    async def get_turbopage(self, client_login: str, turbo_page_id: int) -> TurboPageGetItem:
        result_class: TurboPages.result = await self._make_request(
            {'SelectionCriteria': {'Ids': [turbo_page_id]}}, TurboPages, client_login)
        if result_class.TurboPages is not None and len(result_class.TurboPages) == 1:
            return result_class.TurboPages[0]
        logger.debug('DirectKeyError: result_class.TurboPages is None: %s.', str(result_class.TurboPages is None))
        raise DirectKeyError('Error: Wrong TurboPage Id!')

    async def get_keywords(self, client_login: str, campaign_id: int) -> [KeywordGetItem]:
        result_class: Keywords.result = await self._make_request(
            {'SelectionCriteria': {'CampaignIds': [campaign_id]}}, Keywords, client_login)
        return result_class.Keywords

    async def get_bidmodifiers(self, client_login: str, campaign_id: int) -> [BidModifierGetItem]:
        result_class: BidModifiers.result = await self._make_request(
            {'SelectionCriteria': {'CampaignIds': [campaign_id]}}, BidModifiers, client_login)
        return result_class.BidModifiers

    async def get_audiencetargets(self, client_login: str, campaign_id: int) -> [AudienceTargetGetItem]:
        result_class: AudienceTargets.result = await self._make_request(
            {'SelectionCriteria': {'CampaignIds': [campaign_id]}}, AudienceTargets, client_login)
        return result_class.AudienceTargets

    async def get_client(self, client_login: str) -> ClientGetItem:
        try:
            result_class: Clients.result = await self._make_request({}, Clients, client_login)
            return result_class.Clients[0]
        except DirectAPIError as err:
            logger.debug('DirectKeyError: in get_client: %s', str(err))
            raise DirectKeyError(f'Error: Probably wrong client login or DirectAPIError, check logs. {str(err)}')
