from tvm2 import TVM2
from aiohttp import web
from dataclasses import dataclass
from aiohttp.web import Request, middleware
from crm.supskills.common.deploy_logger import create_logger

logger = create_logger('tvm_logger')


class TVMError(Exception):
    pass


class TvmClient:
    async def parse_service_ticket(self, ticket):
        ...

    async def parse_user_ticket(self, ticket):
        ...

    async def get_service_ticket(self, destination) -> str:
        ...


def get_tvm_client(config):
    return TVM2(client_id=config.tvm2_client_id,
                blackbox_client=config.tvm2_blackbox,
                allowed_clients=config.tvm2_allowed_client_ids,
                destinations=config.tvm2_allowed_client_ids)


@dataclass
@middleware
class TvmServiceMiddleware:
    tvm_client: TvmClient
    tvm2_active: bool

    # Check request header for right service ticket
    async def __call__(self, request: Request, handler):
        if request.match_info.route.name in {'ping'}:
            return await handler(request)
        if not self.tvm2_active:
            logger.info('TVM is disabled.')
            return await handler(request)
        ticket = request.headers.get('X-Ya-Service-Ticket')
        if ticket:
            try:
                parsed_ticket = await self.tvm_client.parse_service_ticket(ticket=ticket)
                if parsed_ticket:
                    logger.info('Right ticket.')
                    return await handler(request)
                else:
                    logger.error('Error 403: Forbidden, wrong service ticket!')
                    return web.json_response(status=403, text='Forbidden, wrong service ticket!')
            except Exception as e:
                logger.debug('TVM Error: %s', str(e))
                raise TVMError(e)
        else:
            logger.info('Ticket is None')
        logger.error('Error 401: Unauthorized, service ticket not found!')
        return web.json_response(status=401, text='Unauthorized, service ticket not found!')
