import __res as res
import importlib
import inspect

from crm.supskills.common.deploy_logger import create_logger
from crm.supskills.direct_skill.src.core.models.button import Button
from crm.supskills.direct_skill.src.core.models.intent import Intent
from crm.supskills.direct_skill.src.intents.general.ask_topic import AskTopic
from crm.supskills.direct_skill.src.consts.scenario.buttons.ask_topic import ask_topic_buttons
from crm.supskills.direct_skill.src.intents.general.simple_direct_intent import SimpleDirectIntent

logger = create_logger('selection_logger')


class IntentStorage:
    def __init__(self, max_count_of_intents_in_request, bunker, direct5_client, direc4_client):
        self.max_count_of_intents_in_request = max_count_of_intents_in_request
        self._path = 'crm.supskills.direct_skill.src.intents'
        self.intents = {}
        self.bunker = bunker
        self.direct5_client = direct5_client
        self.direct4_client = direc4_client

    def _get_modules(self):
        modules = []

        init_suffix = '.__init__'

        for name in res.iter_py_modules():
            if name.startswith(self._path):
                if name.endswith(init_suffix):
                    modules.append(importlib.import_module(name))
                    name = name[:-len(init_suffix)]

                modules.append(importlib.import_module(name))
        return modules

    @staticmethod
    def _load_classes(module):
        classes = []

        for _, cls in inspect.getmembers(module, inspect.isclass):
            if issubclass(cls, Intent) and cls != Intent:
                classes.append(cls)
        return classes

    def _get_intents(self):
        if not self.intents:

            for m in self._get_modules():
                for cls in self._load_classes(m):
                    cls_instance = cls(self.bunker)
                    self.intents[cls_instance.name + cls_instance.state] = cls_instance

    def get_intent_instance(self, intent: str, state='') -> Intent:
        """ Get correct class from intents dictionary based on intent name and state. """
        intent_with_state = self.intents.get(intent + state)

        if intent_with_state:
            return intent_with_state

        intent_without_state = self.intents.get(intent)

        if intent_without_state:
            return intent_without_state

        return self.intents.get('call_operator')

    async def pull_buttons(self, intents_found) -> (list, list):
        buttons, intents = [], []
        for intent in intents_found:
            button_text: str = await self.bunker.get_node(*ask_topic_buttons.get_button(intent), skip_if_no_var=True)
            if button_text:
                buttons.append(Button(button_text))
                intents.append(intent)
        return buttons, intents

    async def select_intent(self, conversation, *args, **kwargs):
        is_new = conversation.request['session']['new']
        intents = list(filter(lambda i: '.' not in i, conversation.request['request']['nlu']['intents'].keys()))
        general_intents = [i for i in intents if '_general_topic' in i]
        specific_intents = [i for i in intents if '_general_topic' not in i]
        general_buttons, general_filtered_intents = await self.pull_buttons(general_intents)
        specific_buttons, specific_filtered_intents = await self.pull_buttons(specific_intents)

        self._get_intents()

        if len(intents) == 0:
            if is_new:
                intent_name = 'empty_intent'
            else:
                intent_name = 'call_operator'
        elif len(specific_intents) == 1:
            intent_name = specific_intents[0]
        elif len(specific_intents) == 0 and len(general_intents) == 1:
            intent_name = general_intents[0]
        elif len(specific_buttons) > self.max_count_of_intents_in_request:
            intent_name = 'call_operator'
        elif len(specific_filtered_intents) == 0 and len(general_filtered_intents) > 1:
            intent_name = 'ask_topic'
        elif len(specific_filtered_intents) > 1:
            intent_name = 'ask_topic'
        elif len(specific_filtered_intents) == 1:
            intent_name = specific_intents[0]
        elif len(specific_filtered_intents) == 0 and len(general_filtered_intents) == 1:
            intent_name = general_intents[0]
        else:
            intent_name = 'call_operator'

        intent = self.get_intent_instance(intent_name, conversation.user_state['state'] or '')
        params = None
        logger.debug(intent.name)

        if isinstance(intent, AskTopic):
            logger.debug('AskTopic, len(buttons) = %d', len(general_buttons) + len(specific_buttons))
            params = specific_buttons or general_buttons
            return intent, params

        if isinstance(intent, SimpleDirectIntent):

            intent_name, state_name = await intent.check_direct_data(conversation,
                                                                     self.direct5_client,
                                                                     self.direct4_client)
            intent = self.get_intent_instance(intent_name, state_name)
            logger.debug(intent.name)

        return intent, params
