from aiohttp import web
from crm.supskills.common.deploy_logger import create_logger
from crm.supskills.direct_skill.src.core.models.conversation import Conversation
from crm.supskills.direct_skill.src.core.models.simple_intent import SimpleIntent
from crm.supskills.direct_skill.src.core.schemas.request import FinalRequestSchema
from crm.supskills.direct_skill.src.core.schemas.response import FinalResponseSchema
from crm.supskills.direct_skill.src.intent_storage import IntentStorage
from ylog.context import log_context


logger = create_logger('dialog_logger')


class DialogHandler:

    def __init__(self, max_count_of_intents_in_request, bunker, direct5_client, direc4_client):
        super(DialogHandler, self).__init__()
        self.intent_storage = IntentStorage(max_count_of_intents_in_request, bunker, direct5_client, direc4_client)
        self.request_data = dict()
        self.session = dict(state='', empty_intents_count=0)
        self.response = dict()
        self.intent: SimpleIntent = None

    async def post(self, request):
        request_schema = FinalRequestSchema()
        response_schema = FinalResponseSchema()
        awaited_request = await request.json()
        self.request_data = request_schema.dump(awaited_request)

        operator_chat_id = self.get_user_identifier('operator_chat_id')

        with log_context(chat_id=operator_chat_id):
            logger.info(self.request_data)

            self._form_session()
            self._form_response()
            await self._handle_conversation()
            await self._form_analytics()

            processed_response = response_schema.dump(self.response)
            logger.info(processed_response)

            return web.json_response(processed_response)

    def _form_session(self):
        saved_session = self.request_data['state'].get('session')
        if saved_session:
            self.session = saved_session

    def _form_response(self):
        self.response = dict(
            version=self.request_data['version'],
            session=self.request_data['session'],
            session_state=self.session,
            response=dict()
        )

    async def _form_analytics(self) -> None:
        """ Collect data on received intents and modify self.response.analytics. """

        category_intents = await self.intent_storage.bunker.get_all_nodes_in_path('category_matching')

        intents_set = set(self.request_data['request']['nlu']['intents'].keys())
        intents_ids = {key: category_intents[key] for key in category_intents.keys() & intents_set}

        intent_category = intents_ids[self.intent.name] if intents_ids.get(self.intent.name) else 0

        self.response['analytics'] = \
            {
                'events': [
                    {
                        'name': 'intents',
                        'value': intents_ids
                    },
                    {
                        'name': 'selected_intent',
                        'value': {
                            self.intent.name: intent_category
                        }
                    },
                    {
                        'name': 'intent_state',
                        'value': {
                            'intent_state': self.intent.state or None
                        }
                    }
                ]
            }

    async def _handle_conversation(self) -> None:
        """ Modify self.response according to selected intent. """

        # response has a response dict inside that is modified with what we want to send to our user
        response_in_response = self.response['response']

        conversation = Conversation(self.request_data, response_in_response, self.session)
        self.intent, params = await self.intent_storage.select_intent(conversation)
        await self.intent.act(conversation, params)

    def get_user_identifier(self, identifier: str) -> str:
        """ Get value for 'login', 'puid' from self.request"""

        user_identifier = None
        floyd_user = self.request_data['session'].get('floyd_user')

        if floyd_user:
            user_identifier = floyd_user.get(identifier)

        return user_identifier
