const { lookahead } = require('../utils/lookahead');
const womBlockGenerator = require('./wom-block-generator');
const { inlinePairedText } = require('./inline-paired-text');

const { isWS } = require('../utils/char-test');

const injectBefore = (target, beforeObject, objects) => {
    target.splice(target.indexOf(beforeObject), 0, ...objects);
};

function eatDefinitionTitle(ctx) {
    const { index, value } = ctx;

    const closePos = value.indexOf('?)', index);
    const eqPos = value.indexOf('==', index);
    const hasEq = eqPos !== -1 && closePos > eqPos;

    const title = value.slice(index, hasEq ? eqPos : lookahead(value, index, isWS));

    ctx.cut(title.length + (hasEq ? 2 : 1));

    return {
        title,
        equals: hasEq,
    };
}

const { patch: decodePatch } = require('./decode-patch');

const womHeading = require('../tokenizers/heading-wom');
const womStaff = require('../tokenizers/staff-wom');
const womLink = require('../tokenizers/link-wom');
const womImage = require('../tokenizers/image-wom');
const womTicket = require('../tokenizers/ticket-wom');
const womBreak = require('../tokenizers/break-wom');
const womEscapeTilde = require('../tokenizers/escape-tilde-wom');
const womEscape = require('../tokenizers/escape-wom');
const womColor = require('../tokenizers/color-wom');
const womRemark = require('../tokenizers/remark-wom');
const { womHtml } = require('../tokenizers/html-wom');
const { womFormatter } = require('../tokenizers/formatter-wom');
const { womBlock } = require('../tokenizers/block-wom');
const { womMarkdown } = require('../tokenizers/markdown-wom');
const { womBlockquote } = require('../tokenizers/blockquote-wom');
const { womCut } = require('../tokenizers/cut-wom');
const { womTable } = require('../tokenizers/table-wom');
const { womAction } = require('../tokenizers/action-wom');
const paragraph = require('../tokenizers/paragraph');

const patchedUrl = require('../tokenizers/url');
const list = require('../tokenizers/list');
const breakLocator = require('./locate/break').locator;

function plugin(options = {}) {
    this.Parser = decodePatch(this.Parser);
    this.Parser.prototype.womOptions = options;

    const actions = { all: [] };

    if (options.actions) {
        for (const { name } of options.actions) {
            actions.all.push(name);
        }
    }

    const {
        inlineTokenizers,
        inlineMethods,
        blockTokenizers,
        blockMethods,
        interruptParagraph,
        interruptList,
    } = this.Parser.prototype;

    inlineTokenizers.break.locator = breakLocator;

    const womDefinition = womBlockGenerator('womDefinition', '(?', '?)', {
        eatFirst: eatDefinitionTitle,
        inline: true,
    });

    const myInlineTokenizers = new Map([
        ['womBreak', womBreak],

        ['womItalic', inlinePairedText(['/', 2, false], 'womItalic')],
        ['womUnderline', inlinePairedText(['_', 2, false], 'womUnderline')],
        ['womMonospace', inlinePairedText(['#', 2, false], 'womMonospace')],
        ['womSmall', inlinePairedText(['+', 2, false], 'womSmall')],
        ['womStrike', inlinePairedText(['-', 2, false], 'womStrike')],
        ['strong', inlinePairedText(['*', 2, false], 'strong')],

        // Тут меняется базовый emphasis на "свой"
        // В базовом отличается поведение '_' от '*' и нам это все портит
        // Мы так не можем костылять баги ремарка: https://st.yandex-team.ru/WIKI-12414
        ['emphasis-asterisk', inlinePairedText(['*', 1, false], 'emphasis')],
        ['emphasis-underscore', inlinePairedText(['_', 1, true], 'emphasis')],

        ['womSuperscript', inlinePairedText(['^', 2, false], 'womSuperscript')],
        ['womSubscript', inlinePairedText(['v', 2, false], 'womSubscript')],

        ['womQuestion', inlinePairedText(['?', 2, false], 'womQuestion')],
        ['womRemark', womRemark],

        ['womLink', womLink],

        ['womImage', womImage],

        ['womHtml', womHtml({ inline: true })],
        ['womMarkdown', womMarkdown({ inline: true })],
        ['womFormatter', womFormatter({ inline: true })],
        ['womBlock', womBlock({ inline: true })],
        ['womBlockquote', womBlockquote({ inline: true })],
        ['womDefinition', womDefinition],
        ['womCut', womCut({ inline: true })],

        ['womEscape', womEscape],

        ['womColor', womColor],

        ['womStaff', womStaff],

        ['womTable', womTable({ inline: true })],
    ]);

    if (options.actions) {
        myInlineTokenizers.set('womAction', womAction({
            inline: true,
            names: actions.all,
        }));
    }

    delete inlineTokenizers.emphasis;

    for (const [key, fn] of myInlineTokenizers) {
        inlineTokenizers[key] = fn;
    }

    inlineMethods.unshift(...myInlineTokenizers.keys());

    // NOTE: Эта штука конфликтует со ~~strike~~
    // Пушим ее в самый конец, но перед `text`
    inlineTokenizers.womEscapeTilde = womEscapeTilde;
    injectBefore(inlineMethods, 'text', ['womEscapeTilde']);

    if (options.tracker) {
        // Мы проверяем ссылку на трекер, поэтому он должен быть перед токеном url
        inlineTokenizers.womTicket = womTicket;
        injectBefore(inlineMethods, 'url', ['womTicket']);
    }

    inlineTokenizers.url = patchedUrl;

    const myBlockTokenizers = new Map([
        ['womHtml', womHtml({ inline: false })],
        ['womMarkdown', womMarkdown({ inline: false })],
        ['womFormatter', womFormatter({ inline: false })],
        ['womBlock', womBlock({ inline: false })],
        ['womBlockquote', womBlockquote({ inline: false })],
        ['womCut', womCut({ inline: false })],

        ['womTable', womTable({ inline: false })],

        ['womHeading', womHeading],
    ]);

    if (options.actions) {
        myBlockTokenizers.set('womAction', womAction({
            inline: false,
            names: actions.all,
        }));
    }

    for (const [key, fn] of myBlockTokenizers) {
        blockTokenizers[key] = fn;
    }

    blockTokenizers.list = list;
    blockTokenizers.paragraph = paragraph;

    injectBefore(blockMethods, 'setextHeading', myBlockTokenizers.keys());

    // Любой блок может порвать параграф,
    // не перечисленные ниже блоки проверяются другим механизмом
    interruptParagraph.push(
        ['womHeading'],
        ['list'],
    );

    // Любой блок, кроме самого списка может порвать список,
    // не перечисленные ниже блоки проверяются другим механизмом
    interruptList.push(
        ['womHeading'],
    );

    if (options.actions) {
        interruptParagraph.push(
            ['womAction'],
        );
        interruptList.push(
            ['womAction'],
        );
    }
}

module.exports = plugin;
