const { lookahead } = require('../utils/lookahead');
const { isWS } = require('../utils/char-test');
const { shiftLocation } = require('../lib/tokenize-skip-node/utils');
const isNotWS = char => !isWS(char);
const isLineFeed = char => char === '\n';
const isEqual = char => char === '=';
const isPlusSign = char => char === '+';
const isNotEqual = char => !isEqual(char);

const MAX_DEPTH = 6;

function womHeading(eat, value, silent) {
    const now = eat.now();

    let index = 0;
    let length = 0;
    let nextIndex = 0;

    let subvalue = '';
    let content = '';
    let queue = '';

    /* Eat initial spacing. */
    index = lookahead(value, index, isNotWS);

    /* Eat equals. */
    nextIndex = lookahead(value, index, isNotEqual);
    queue = value.slice(index, nextIndex);
    const depth = nextIndex - index - 1;
    index = nextIndex;

    let char = value[index];

    if (depth < 1 || depth > MAX_DEPTH || isEqual(char)) {
        return;
    }

    // ==+
    let expandable = false;

    if (isPlusSign(char)) {
        expandable = true;
        index += 1;
        char = value[index];
    }

    /*
    * Examples:
    * =() ...
    * =+() ...
    * =()+ ...
    */
    let anchor = null;
    if (char === '(' && value[index + 1] !== '(') {
        nextIndex = value.indexOf(')', index + 1);

        if (nextIndex !== -1) {
            anchor = value.slice(index + 1, nextIndex);
            index = nextIndex + 1;
            char = value[index];
        }
    }

    // Check for expandable again couse it can be `=()+ ...`
    if (anchor && !expandable) {
        if (isPlusSign(char)) {
            expandable = true;
            index += 1;
            char = value[index];
        }
    }

    /* Exit when not in pedantic mode without spacing. */
    if (queue.length === 0 && !isLineFeed(char)) {
        return;
    }

    if (silent) {
        return true;
    }

    /* Eat content. */
    const innerFirstIndex = index;

    subvalue += value.slice(0, innerFirstIndex);
    queue = '';
    content = '';
    length = value.length;

    while (index < length) {
        char = value[index];

        if (isLineFeed(char)) {
            // Если строка обрывается, значит сбор закончен
            break;
        }

        if (queue === '' && !isEqual(char)) {
            // Собираем контент
            content += char;
            index += 1;
            continue;
        }

        if (queue !== '' && !isEqual(char) && !isWS(char)) {
            // Очередь не пустая, но встречаем символ,
            // отличный от свисающего `=+\s*`
            content += queue + char;
            queue = '';
            index += 1;
            continue;
        }

        queue += char;
        index += 1;
    }

    subvalue += content + queue;

    let inner = content;

    now.column += innerFirstIndex;
    now.offset += innerFirstIndex;

    inner = inner.trimLeft();

    shiftLocation(now, content, 0, content.length - inner.length);

    inner = inner.trimRight();

    return eat(subvalue)({
        type: 'womHeading',
        depth,
        anchor,
        expandable,
        children: this.tokenizeInline(inner, now),
    });
}

module.exports = womHeading;
