const { isWS } = require('../utils/char-test');

const validLoginCharacterRe = /[-a-zа-яё0-9._]/i;
const wordRe = /[\wа-яё]/i;

const isValidCharacter = function isValidCharacter(ch) {
    validLoginCharacterRe.lastIndex = 0;
    return validLoginCharacterRe.test(ch);
};
const isNotValidCharacter = char => !isValidCharacter(char);
const isNotWS = char => !isWS(char);
const isDot = char => char === '.';
const isNotADot = char => !isDot(char);
const isW = function isW(ch) {
    wordRe.lastIndex = 0;
    return wordRe.test(ch);
};

const STAFF_PREFIXES = ['кто:', 'кого:', 'укого:', 'ского:', 'кому:', 'кем:', 'ком:', 'оком:', 'staff:', '@'];
const CLUB_PREFIXES = ['club:', 'клуб:'];
const VALID_PREFIXES = [].concat(CLUB_PREFIXES, STAFF_PREFIXES);
const STAFF_SUFFIXES = ['@'];

const VALID_PREFIXES_FIRST_LETTERS = new Set(VALID_PREFIXES.map(p => p.charAt(0)));

/**
 * Truthy variants
 * ? — vasya@
 * '@' — @vasya
 * 'к' — кто:vasya, кому:vasya, кого:vasya, кем:vasya, клуб:классный
 * 'у' — укого:vasya
 * 'о' — оком:vasya
 * 's' — staff:vasya
 * 'c' — club:awesome
 * @param {String} ch — Character
 * @returns {Boolean}
 */
const isValidLinkProbably = ch => isValidCharacter(ch) || VALID_PREFIXES_FIRST_LETTERS.has(ch);

function lookAhead(value, test, offset) {
    const l = value.length;
    let i = offset;
    while (i < l && !test(value.charAt(i))) {
        i += 1;
    }
    return i;
}

function lookBehind(value, test, offset) {
    let i = offset;
    while (i > 0 && !test(value.charAt(i - 1))) {
        i -= 1;
    }
    return i;
}

function womStaff(eat, value, silent) {
    let index = womStaff.locator(value, 0);
    if (index !== 0) {
        return false;
    }

    index = lookAhead(value, isNotWS, 0);
    {
        // Early return for absolutely incorrect
        let character = value.charAt(index);
        if (!isValidLinkProbably(character)) {
            return;
        }
    }

    let lastIndex = Infinity;
    let staff = null;
    const props = {
        case: null,
        at: null,
    };

    // @name, staff:name, кто:name, etc:name
    const prefix = VALID_PREFIXES.find(p => value.indexOf(p, index) === index);
    if (prefix) {
        lastIndex = lookAhead(value, isNotValidCharacter, index + prefix.length);

        // точка не может находится в конце логина
        const pos = lookBehind(value, isNotADot, lastIndex);
        if (pos >= index + prefix.length) {
            lastIndex = pos;
        }

        if (!isW(value.charAt(lastIndex))) {
            prefix === '@' ?
                (props.at = 'prefix') :
                (props.case = prefix.slice(0, -1));
            staff = value.slice(index + prefix.length, lastIndex);
        }
    }

    // name@
    if (!staff) {
        lastIndex = lookAhead(value, isNotValidCharacter, index);
        if (lastIndex !== -1 && value.charAt(lastIndex) === '@' && !isW(value.charAt(lastIndex + 1))) {
            props.at = 'suffix';
            staff = value.slice(index, lastIndex);
            lastIndex += 1;
        }
    }

    if (!staff) {
        return false;
    }
    if (silent) {
        return true;
    }

    return eat(value.slice(0, lastIndex))({
        type: CLUB_PREFIXES.includes(prefix) ? 'womClub' : 'womStaff',
        value: staff,
        ...props,
    });
}

womStaff.locator = function staffLocator(value, fromIndex) {
    // Ранний выход если не найдено потенциальных стафф блоков
    if (value.indexOf('@', fromIndex) === -1 && value.indexOf(':', fromIndex) === -1) {
        return -1;
    }

    const results = [];

    // Проходим по всем префиксам и если находим совпадения записываем в результат
    for (let i = 0; i < VALID_PREFIXES.length; i++) {
        const prefix = VALID_PREFIXES[i];
        let pos = value.indexOf(prefix, fromIndex);
        if (pos === -1 || isW(value.charAt(pos - 1))) {
            continue;
        }

        results.push(pos);
    }

    // Проходим по всем суффксам, и проверяем, что ему предшествует логин пользователя
    for (let i = 0; i < STAFF_SUFFIXES.length; i++) {
        const suffix = STAFF_SUFFIXES[i];
        let pos = value.indexOf(suffix, fromIndex);
        const nextChar = value.charAt(pos + 1);
        if (pos === -1 || isValidCharacter(nextChar) && !isDot(nextChar)) {
            continue;
        }

        pos = lookBehind(value, isNotValidCharacter, pos);
        if (isW(value.charAt(pos - 1))) {
            continue;
        }

        results.push(pos);
    }

    if (results.length === 0) {
        return -1;
    }

    const idx = Math.min.apply(null, results);

    return idx;
};

module.exports = womStaff;
