const whitespace = require('is-whitespace-character');
const urlJoin = require('url-join');

const { eatUrl } = require('../lib/eat-url');
const {
    CH_SLASH, CH_QUESTION, CH_HASH, CH_LEFT_SQUARE_BRACKET,
} = require('../utils/charcodes');

const lineFeed = '\n';

const ticketRE = /(?:^|\s)(?:https?:\/\/.+?)?([A-Z]{2,20}-[0-9]+)(?:$|\b)/g;
const ticketDescriptionRe = /\[ .+? ]\( .*? \)/;

function womTicket(eat, value, silent) {
    /*
     * Парсит тикеты в трекере
     * Должен идти перед автолинками, так как парсит ссылки
     * Шаблон: A-Z{2,15}-\d{1,}
     * Возможные варианты:
     *  - KEY-123
     *  - https://tracker.ru/KEY-123
     *  - Оба варианта + hash комментария
     *  - Оба варианта + [ описание ]( автор )
     */

    let index = 0;
    let lineIndex = value.indexOf(lineFeed, index);
    if (lineIndex === -1) {
        lineIndex = value.length;
    }
    // Внутри ссылки на тикет не может быть переноса строк
    let subValue = value.substring(index, lineIndex);

    ticketRE.lastIndex = null;
    const match = ticketRE.exec(subValue);

    // Не нашли признаков тикета, выходим
    if (!match) {
        return;
    }

    index = match.index;

    // Смотрим что если перед тикетом/урлом был ws смещаемся на 1
    if (whitespace(value.charCodeAt(index))) {
        index++;
    }

    // match[0] - полное совпадение вместе с урлом https://tracker.ru/QQQ-123
    // match[1] - совпадение только группы с тикетом QQQ-123
    // считаем индексы начала и конца относительно всей входной строки
    const ticketValue = match[1];
    const ticketIndex = index + match[0].indexOf(ticketValue);
    const ticketEndIndex = ticketIndex + ticketValue.length;

    let ticketUrl = '';
    let ticketParams = '';
    // Проверяем является ли тикет частью конструкции https://tracker.ru/QQQ-123
    if (ticketIndex !== 0) {
        // Если нет слеша перед тикетом, это точно не чатсь урла, выходим
        if (subValue.charCodeAt(ticketIndex - 1) !== CH_SLASH) {
            return;
        }

        // Список урлов трекера передается в настройках, основной урл + алиасы
        const urls = [this.womOptions.tracker.url].concat(this.womOptions.tracker.aliases);
        const currentUrl = subValue.substring(0, ticketIndex - 1);
        for (let i = 0; i < urls.length; i++) {
            const trackerUrl = urls[i];
            if (trackerUrl === currentUrl) {
                ticketUrl = currentUrl + '/';
                break;
            }
        }

        // Если тикет не в начале и это не часть урла на трекер сразу выходим
        if (!ticketUrl) {
            return;
        }

        // Тут мы убедились, что надо съесть ссылку целиком
        index = ticketEndIndex;

        // Если ссылка передана тикетом, то дополнительно могут быть переданы get параметры или хеш
        // пропускаем / в конце урла
        if (subValue.charCodeAt(index) === CH_SLASH) {
            index++;
        }

        let nextChar = subValue.charCodeAt(index);
        // Параметры могут быть только # или ?
        if (nextChar === CH_HASH || nextChar === CH_QUESTION) {
            // eat params
            const [newIndex, params] = eatUrl(index, subValue);
            if (params.length) {
                index = newIndex;
                ticketParams = params;
            }
        } else if (!isNaN(nextChar) && !whitespace(nextChar) && nextChar !== CH_LEFT_SQUARE_BRACKET) {
            // Если после тикета в урле есть что угодно кроме парамтеров выходим
            return;
        } else if (nextChar === CH_LEFT_SQUARE_BRACKET) {
            // Если после ссылки стоит открывающий брекет, проверяем, что это описание, иначе выходим
            ticketDescriptionRe.lastIndex = index;
            const matchDescription = ticketDescriptionRe.exec(subValue);
            if (!matchDescription) {
                return;
            }
        }
    } else {
        // Так как мы съели тикет, смещаем индекс в его конец
        index = ticketEndIndex;
    }

    if (silent) {
        return true;
    }

    const node = {
        type: 'womTicket',
        value: ticketValue,
        url: urlJoin(this.womOptions.tracker.url, ticketValue),
    };

    if (ticketUrl) {
        node.url = ticketUrl + ticketValue + ticketParams;
    }

    // TODO: удалить этот кусок когда оторвем описание тикетов
    // Проверка на разложенные бекендом тикеты "KEY-123[ summary ]( author )"
    ticketDescriptionRe.lastIndex = index;
    const matchDescription = ticketDescriptionRe.exec(subValue);
    if (matchDescription) {
        // Полностью игнорируем описание тикета, так как будем это отрывать
        index += matchDescription[0].length;
    }

    return eat(value.slice(0, index))(node);
}

// Locator
womTicket.locator = (value, fromIndex) => {
    // `QUEUE-12345`
    let match;
    // Используем -1, так как ^ не считается началом строки относительно индекса,
    // и проверка будет относительно \s
    ticketRE.lastIndex = fromIndex - 1;

    match = ticketRE.exec(value);

    if (!match) {
        return -1;
    }

    let index = match.index;

    // Смотрим что если перед тикетом/урлом был ws смещаемся на 1
    if (whitespace(value.charCodeAt(index))) {
        index++;
    }

    if (index < fromIndex) {
        return -1;
    }

    return index;
};

womTicket.notInLink = true;

module.exports = womTicket;
